{ ------------------------------------------------------------------- }
{ Dieser Quelltext ist urheberrechtlich geschuetzt.                   }
{ (c) 1991-1999 Peter Mandrella                                       }
{ (c) 2000-2001 OpenXP-Team                                           }
{ (c) 2002-2024 FreeXP, http://www.freexp.de                          }
{ CrossPoint ist eine eingetragene Marke von Peter Mandrella.         }
{                                                                     }
{ Die Nutzungsbedingungen fuer diesen Quelltext finden Sie in der     }
{ Datei SLIZENZ.TXT oder auf http://www.crosspoint.de/oldlicense.html }
{ ------------------------------------------------------------------- }

{ CrossPoint - Zeichensatzdecodierung und -konvertierung }

{$I XPDEFINE.INC }
{$O+,F+}

unit mimedec;

interface

uses xpglobal,typeform;

type LongHdrT  = array[1..65500] of char;     { Langer Header > 253 Zeichen }
     LongHdrP  = ^LongHdrT;

const
  cs_us_ascii   =    0;  { US-ASCII                         }
  cs_cp_437     =  437;  { CP437        (DOS)               }
  cs_cp_850     =  850;  { CP850        (DOS)               }
  cs_cp_858     =  858;  { CP858        (DOS, mit Euro)     }
  cs_iso8859_1  =    1;  { ISO-8859-1   (Latin1)            }
  cs_iso8859_2  =    2;  { ISO-8859-2   (Latin2)            }
  cs_iso8859_3  =    3;  { ISO-8859-3   (Latin3)            }
  cs_iso8859_4  =    4;  { ISO-8859-4   (Latin4)            }
  cs_iso8859_9  =    9;  { ISO-8859-9   (Latin5)            }
  cs_iso8859_10 =   10;  { ISO-8859-10  (Latin6)            }
  cs_iso8859_13 =   13;  { ISO-8859-13  (Latin7)            }
  cs_iso8859_14 =   14;  { ISO-8859-14  (Latin8)            }
  cs_iso8859_15 =   15;  { ISO-8859-15  (Latin9,  mit Euro) }
  cs_iso8859_16 =   16;  { ISO-8859-16  (Latin10, mit Euro) }
  cs_win1250    = 1250;  { Windows-1250 (mit Euro)          }
  cs_win1252    = 1252;  { Windows-1252 (mit Euro)          }
  cs_win1254    = 1254;  { Windows-1254 (mit Euro)          }
  cs_win1257    = 1257;  { Windows-1257 (mit Euro)          }
  cs_macroman   = 2000;  { Mac OS Roman (mit Euro)          }
  cs_utf_7      = 7000;  { UTF-7        (Unicode, mit Euro) }
  cs_utf_8      = 8000;  { UTF-8        (Unicode, mit Euro) }
  { ----------------------------------------------------------------------- }
  rfcMime      : boolean = false; { MIME-Decodierung strikt nach RFC2047 (UUZ) }
  maxUTFLen    : word    = 248;   { max. an 'UTFxToIBM' zu bergebende Lnge }
  start_of_UTF : boolean = false; { 'sc_rest' bei UTF-Decodierung nicht ignorieren }
  end_of_UTF   : boolean = false; { 'sc_rest' bei UTF-Decodierung belegen          }
  fcs          : string[30] = ''; { Name Fallback-Charset }
  cs_fallback  : boolean = false; { Charset-Fallback }
  { ----------------------------------------------------------------------- }


{ -------------------------------------------------------- }
{ Quellen fr Charset-Tabellen und -Informationen:         }
{ -------------------------------------------------------- }
{ http://www.iana.org/assignments/character-sets           }
{ http://czyborra.com/charsets/iso8859.html                }
{ http://www.evertype.com                                  }
{ http://www.microsoft.com/globaldev/reference/cphome.mspx }
{                                                          }
{ [... to be continued ...]                                }
{                                                          }
{ -------------------------------------------------------- }


{ ausgehend: #46 (".") = unkonvertierbar }

const IBM2ISOtab : array[0..255] of byte =  {EUR #238 siehe set_IBM2ISOtab}
{000} (  32, 46, 46, 46, 46, 46, 46, 42, 46,  9, 10, 46, 12, 13, 46, 42,
{016}    62, 60, 46, 33,182,167, 95, 46, 46, 46, 62, 60, 45, 46, 46, 46,
{032}    32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
{048}    48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
{064}    64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
{080}    80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
{096}    96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
{112}   112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,

{128}   199,252,233,226,228,224,229,231,234,235,232,239,238,236,196,197,
{144}   201,230,198,244,246,242,251,249,255,214,220,162,163,165, 46, 46,
{160}   225,237,243,250,241,209,170,186,191, 45,172,189,188,161,171,187,
{176}    35, 35, 35,124, 43, 43, 43, 43, 43, 43,124, 43, 43, 43, 43, 43,
{192}    43, 43, 43, 43, 45, 43, 43, 43, 43, 43, 43, 43, 43, 45, 43, 43,
{208}    43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 35, 35, 35, 35, 35,
{224}    97,223, 71,112, 83,115,181,116, 80, 84, 79,100, 46,102,101, 46,
{240}    61,177, 62, 60,124,124,247, 61,176,183,183, 46,110,178,183, 32);


{ eingehend: #177 ("") = unkonvertierbar }

     ISO1_2IBMtab : array[128..255] of byte =      { oft: #128 = Euro }
{128} (euro,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32,173,155,156,177,157,124, 21, 34, 67,166,174,170, 45, 82,196,
{176}   248,241,253, 51, 39,230, 20,250, 44, 49,167,175,172,171,177,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 84,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   116,164,149,162,147,111,148,246,148,151,163,150,129,121,116,152);


{ ISO-8859-1 (West-Europa), alias "Latin 1", oft falsch: #128 = Euro }
{ Windows-1252 (West-Europa), alias "Win Latin 1", #128 = Euro }
      WIN1252_2IBMtab : array[128..255] of byte =
{128} (euro, 32, 39,159, 34,177,177,177, 94,177, 83, 60, 79, 32, 90, 32,
{144}    32, 39, 39, 34, 34, 42,196,196,126,177,115, 62,111, 32,122, 89,
{160}    32,173,155,156,177,157,124, 21, 34, 67,166,174,170, 45, 82, 45,
{176}   248,241,253, 51, 39,230, 20,250, 44, 49,167,175,172,171,177,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 84,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   116,164,149,162,147,111,148,246,148,151,163,150,129,121,116,152);

{ ISO-8859-2 (Ost-Europa), alias "Latin 2", kein Euro }
      ISO2_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 65,177, 76,177, 76, 83, 21, 34, 83, 83, 84, 90, 45, 90, 90,
{176}   248, 97,177,108, 39,108,115,177, 44,115,115,116,122, 34,122,122,
{192}    82, 65, 65, 65,142, 76, 67,128, 67,144, 69, 69, 69, 73, 73, 68,
{208}    68, 78, 78, 79, 79,153,153,120, 82, 85, 85,154,154, 89, 84,225,
{224}   114,160,131, 97,132,108, 99,135, 99,130,101,137,101,161,140,100,
{240}   100,110,110,162,147,148,148,246,114,117,163,129,129,121,116,177);

{ ISO-8859-3 (Sd-Europa), alias "Latin 3", kein Euro }
      ISO3_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 72,177,156,177, 32, 72, 21, 34, 73, 83, 71, 74, 45, 32, 90,
{176}   248,104,253, 51, 39,230,104,250, 44,105,115,103,106,171, 32,122,
{192}    65, 65, 65, 32,142, 67, 67,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    32, 78, 79, 79, 79, 71,153,120, 71, 85, 85, 85,154, 85, 83,225,
{224}   133,160,131, 32,132, 99, 99,135,138,130,136,137,141,161,140,139,
{240}    32,164,149,162,147,103,148,246,103,151,163,150,129,117,115,177);

{ ISO-8859-4 (Nord-Europa/Baltisch), alias "Latin 4", kein Euro }
      ISO4_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 65,107, 82,177, 73, 76, 21, 34, 83, 69, 71, 84, 45, 90, 45,
{176}   248, 97,177,114, 39,105,108,177, 44,115,101,103,116, 78,122,110,
{192}    65, 65, 65, 65,142,143,146, 73, 67,144, 69, 69, 69, 73, 73, 73,
{208}    68, 78, 79, 75, 79, 79,153,120,153, 85, 85, 85,154, 85, 85,225,
{224}    97,160,131, 97,132,134,145,105, 99,130,101,137,101,161,140,105,
{240}   100,110,111,107,147,111,148,246,148,117,163,150,129,117,117,177);

{ ISO-8859-9 (Trkisch), alias "Latin 5", kein Euro }
{ Windows-1254 (Trkisch), alias "Win Turkish", #128 = Euro }
      WIN1254_2IBMtab : array[128..255] of byte =
{128} (euro, 32, 39,159, 34,177,177,177, 94,177, 83, 60, 79, 32, 32, 32,
{144}    32, 39, 39, 34, 34, 42,196,196,126,177,115, 62,111, 32, 32, 89,
{160}    32,173,155,156,177,157,124, 21, 34, 67,166,174,170, 45, 82, 45,
{176}   248,241,253, 51, 39,230, 20,250, 44, 49,167,175,172,171,177,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    71,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 73, 83,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   103,164,149,162,147,111,148,246,148,151,163,150,129,105,115,152);

{ ISO-8859-10 (Nord-Europa/Nordisch), alias "Latin 6", kein Euro }
      ISO10_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 65, 69, 71, 73, 73, 75, 21, 76, 68, 83, 84, 90, 45, 85, 78,
{176}   248, 97,101,103,105,105,107,250,108,100,115,116,122,196,117,110,
{192}    65, 65, 65, 65,142,143,146, 73, 67,144, 69, 69, 69, 73, 73, 73,
{208}    84, 78, 79, 79, 79, 79,153, 85,153, 85, 85, 85,154, 89, 84,225,
{224}    97,160,131, 97,132,134,145,105, 99,130,101,137,101,161,140,139,
{240}   116,110,111,162,147,111,148,117,148,117,163,150,129,121,116,107);

{ ISO-8859-13 (Nord-Europa/Baltisch), alias "Latin 7", kein Euro }
      ISO13_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 34,155,156,177, 34,124, 21,153, 67, 82,174,170, 45, 82,146,
{176}   248,241,253, 51, 34,230, 20,250,148, 49,114,175,172,171,177,145,
{192}    65, 73, 65, 67,142,143, 69, 69, 67,144, 90, 69, 71, 75, 73, 76,
{208}    83, 78, 78, 79, 79, 79,153,120, 85, 76, 83, 85,154, 90, 90,225,
{224}    97,105, 97, 99,132,134,101,101, 99,130,122,101,103,107,105,108,
{240}   115,110,110,162,111,111,148,246,117,108,115,117,129,122,122, 39);

{ ISO-8859-14 (West-Europa/Glisch), alias "Latin 8", kein Euro }
      ISO14_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 66, 98,156, 67, 99, 68, 21, 87, 67, 87,100, 89, 45, 82, 89,
{176}    70,102, 71,103, 77,109, 20, 80,119,112,119, 83,121, 87,119,115,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    87, 78, 79, 79, 79, 79,153, 84,153, 85, 85, 85,154, 89, 89,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   119,164,149,162,147,111,148,116,148,151,163,150,129,121,121,152);

{ ISO-8859-15 (West-Europa), alias "Latin 9", alias "Latin 0", #164 = Euro }
      ISO15_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32,173,155,156,euro,157,83, 21,115, 67,166,174,170, 45, 82, 45,
{176}   248,241,253, 51, 90,230, 20,250,122, 49,167,175, 79,111, 89,168,
{192}    65, 65, 65, 65,142,143,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    84,165, 79, 79, 79, 79,153,120,153, 85, 85, 85,154, 89, 84,225,
{224}   133,160,131, 97,132,134,145,135,138,130,136,137,141,161,140,139,
{240}   116,164,149,162,147,111,148,246,148,151,163,150,129,121,116,152);

{ ISO-8859-16 (Ost-Europa/Rumnisch etc.), alias "Latin 10", #164 = Euro }
      ISO16_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
{160}    32, 65, 97, 76,euro,34, 83, 21,115, 67, 83,174, 90, 45,122, 90,
{176}   248,241, 67,108, 90, 34, 20,250,122, 99,115,175, 79,111, 89,122,
{192}    65, 65, 65, 65,142, 67,146,128, 69,144, 69, 69, 73, 73, 73, 73,
{208}    68, 78, 79, 79, 79,153,153, 83,154, 85, 85, 85,154, 69, 84,225,
{224}   133,160,131, 97,132, 99,145,135,138,130,136,137,141,161,140,139,
{240}   100,110,149,162,147,148,148,115,129,151,163,150,129,101,116,152);

{ Windows-1250 (Ost-Europa), alias "Win Latin 2", #128 = Euro }
      WIN1250_2IBMtab : array[128..255] of byte =
{128} (euro, 32, 39, 32, 34,177,177,177, 32,177, 83, 60, 83, 84, 90, 90,
{144}    32, 39, 39, 34, 34, 42,196,196, 32,177,115, 62,115,116,122,122,
{160}    32,177,177, 76,177, 65,124, 21, 34, 67, 83,174,170, 45, 82, 90,
{176}   248,241,177,108, 39,230, 20,250, 44, 97,115,175, 76, 34,108,122,
{192}    82, 65, 65, 65,142, 76, 67,128, 67,144, 69, 69, 69, 73, 73, 68,
{208}    68, 78, 78, 79, 79,153,153,120, 82, 85, 85,154,154, 89, 84,225,
{224}   114,160,131, 97,132,108, 99,135, 99,130,101,137,101,161,140,100,
{240}   100,110,110,162,147,148,148,246,114,117,163,129,129,121,116,177);

{ Windows-1257 (Nord-Europa/Baltisch), alias "Win Baltic", #128 = Euro }
      WIN1257_2IBMtab : array[128..255] of byte =
{128} (euro, 32, 39, 32, 34,177,177,177, 32,177, 32, 60, 32, 34,177, 44,
{144}    32, 39, 39, 34, 34, 42,196,196, 32,177, 32, 62, 32, 45,177, 32,
{160}    32, 32, 69, 71, 73, 32, 75, 21, 76, 68, 83, 84, 90, 45, 85, 78,
{176}   248, 97,101,103, 39,105,107,250,108,100,115,116,122,196,117,110,
{192}    65, 65, 65, 65,142,143,146, 73, 67,144, 69, 69, 69, 73, 73, 73,
{208}    84, 78, 79, 79, 79, 79,153, 85,153, 85, 85, 85,154, 89, 84,225,
{224}    97,160,131, 97,132,134,145,105, 99,130,101,137,101,161,140,139,
{240}   116,110,111,162,147,111,148,117,148,117,163,150,129,121,116,177);

{ Codepage 850 (West-Europa), alias "DOS Latin 1", kein Euro }
      CP850_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,148,156,153,120,159,
{160}   160,161,162,163,164,165,166,167,168, 82,170,171,172,173,174,175,
{176}   176,177,178,179,180, 65, 65, 65, 67,185,186,187,188,155,157,191,
{192}   192,193,194,195,196,197, 97, 65,200,201,202,203,204,205,206,177,
{208}   116, 84, 69, 69, 69,105, 73, 73, 73,217,218,219,220,124, 73,223,
{224}    79,225, 79, 79,111, 79,230,116, 84, 85, 85, 85,121, 89, 45, 39,
{240}    45,241, 95,177, 20, 21,246, 44,248, 34,250, 49, 51,253,254,255);

{ Codepage 858 (West-Europa), alias "DOS Latin 1", #213 = Euro }
      CP858_2IBMtab : array[128..255] of byte =
{128}  (128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
{144}   144,145,146,147,148,149,150,151,152,153,154,148,156,153,120,159,
{160}   160,161,162,163,164,165,166,167,168, 82,170,171,172,173,174,175,
{176}   176,177,178,179,180, 65, 65, 65, 67,185,186,187,188,155,157,191,
{192}   192,193,194,195,196,197, 97, 65,200,201,202,203,204,205,206,177,
{208}   116, 84, 69, 69, 69,euro,73, 73, 73,217,218,219,220,124, 73,223,
{224}    79,225, 79, 79,111, 79,230,116, 84, 85, 85, 85,121, 89, 45, 39,
{240}    45,241, 95,177, 20, 21,246, 44,248, 34,250, 49, 51,253,254,255);

{ Mac OS Roman (West/Nord-Europa, alias "MacRoman", #219 = Euro }
      Mac2IBMtab : array[128..255] of byte =
{128}  (142,143,128,144,165,153,154,160,133,131,132, 97,134,135,130,138,
{144}   136,137,161,141,140,139,164,162,149,147,148,111,163,151,150,129,
{160}   177,248,155,156, 21, 42, 20,225, 82, 67,177, 39, 34,177,146,153,
{176}   236,241,243,242,157,230,177,228,227,227,244,166,167,234,145,148,
{192}   168,173,170,251,159,247,177,174,175,177, 32, 65, 65, 79, 79,111,
{208}   196,196, 34, 34, 39, 39,246,177,152, 89, 47,euro,60, 62,102,102,
{224}   177,250, 39, 34,177, 65, 69, 65, 69, 69, 73, 73, 73, 73, 79, 79,
{240}   177, 79, 85, 85, 85,105, 94,126, 45,177,177,248, 44, 34,177,177);


procedure set_IBM2ISOtab;
procedure IBM2ISO(var s:string; const maxlen:byte; const multichar:boolean);
procedure IBMToIso(var data; var size:word; const multichar:boolean);
procedure IBMToIso1(var data; size:word);
procedure ISO2IBM(var s:string; const charset: word);
procedure IsoToIBM(var data; size:word; const charset:word);
procedure Iso1ToIBM(var data; size:word);
procedure Mac2IBM(var data; size:word);
procedure UTF8ToIBM(var s:string);
procedure UTF7ToIBM(var s:string);
function  RFC_CharsetName(const n:word):string;
function  ZC_CharsetName(s:string):string;
function  ascii_charset(const s:string):boolean;
function  ibm_charset(const s:string):boolean;
function  local_charset(const s:string):boolean;
function  convertible_charset(const s:string):word;
function  iso_charset(s:string):word;
function  supported_charset(const s:string):boolean;
function  any2rfc_charset(const charset:string):string;
function  fallback_charset(const n:word):string;
procedure CharsetToIBM(charset:string; var s:string);


procedure DecodeBase64(var s:string);
function  UnQuotePrintable(var s:string; const eol:byte):boolean;
function  GetHeader(var hdr:LongHdrP; var len:word;
                    var rest:string; var truncated:boolean;
                    const overflow,MIMEdecode,replaceWSP:boolean;
                    const context:byte):string;
procedure RFC2047_Decode(var s:string; const context:byte);
procedure RfcToZcConv(var hdr:LongHdrP; var len:word; const len1,len2:word;
                      var str1,str2,rest:string; var truncated:boolean;
                      const overflow:boolean; context:byte;
                      const emptyfiller:string);
procedure RFC2822_Remove(var s:string; const context:byte);

function posLong(const s:string; const hdr:LongHdrP; frompos:word;
                 const topos:word; const docase:boolean):word;


function MakeMimeBoundary:string;
procedure MimeIsoDecode(var ss:string; maxlen:integer);

implementation  {----------------------------------------------------}

uses xp0,xpovl;


(*
{ my: Testroutine - nicht entfernen! }
function free_stack:word;assembler;
asm
  mov ax,sp;
  add ax,4;
end;
*)


procedure set_IBM2ISOtab;
begin
  if not iso15 then           { ISO1 }
  begin
    IBM2ISOtab[171]:=189;
    IBM2ISOtab[172]:=188;
    if euro_allowed then
      IBM2ISOtab[euro]:=128
    else IBM2ISOtab[euro]:=101;
  end
  else begin                  { ISO15 }
    IBM2ISOtab[171]:=46;
    IBM2ISOtab[172]:=46;
    if euro_allowed then
      IBM2ISOtab[euro]:=164
    else IBM2ISOtab[euro]:=101;
  end;
end;


procedure _IBM2ISOMultiChar(var s:string; const maxlen:byte);
var i,j,k : byte;
        l : integer;
       s2 : string[3];

    Multi_List  : string[15];
    Multi_Konv1 : string[15];
    Multi_Konv2 : string[15];
    Multi_Konv3 : string[15];

begin
  Multi_List:=#26+#27+'';
  Multi_Konv1:='--!ENpPTp/><^';
  Multi_Konv2:='><!SLihhh\==n';
  Multi_Konv3:='---PG--------';  { '-' = '' }
  if iso15 then   { '' und '' konvertieren }
  begin
    Multi_List:=Multi_List + '';
    Multi_Konv1:=Multi_Konv1 + '11';
    Multi_Konv2:=Multi_Konv2 + '//';
    Multi_Konv3:=Multi_Konv3 + '24';
  end;
  i:=1; k:=length(s);
  While i <= k do
  begin
    j:=cpos(s[i],Multi_List);
    if j>0 then
    begin
      s2:=copy(Multi_Konv1,j,1) +
          copy(Multi_Konv2,j,1) +
          iifs(copy(Multi_Konv3,j,1)='-','',copy(Multi_Konv3,j,1));
      l:=i + (length(s2)-1);
      if l>maxlen then  { Wenn's nicht mehr vollstndig in 'maxlen' pat... }
      begin
        s:=left(s,i-1);   { ...String vor dem aktuellen Zeichen abschneiden }
        exit;             { und fertig }
      end
      else begin
        delete(s,i,1);
        insert(s2,s,i);
        l:=k + (length(s2)-1);  { 'inc(k,length(s2)-1)' geht nicht, weil }
        k:=min(l,255);          { 'inc(254,2)' gleich 1 (ergbe Abbruch) }
      end;
      inc(i,length(s2));  { Erstes Zeichen nach Ersetzungszeichen prfen }
    end
    else inc(i);          { Kein Zeichen ersetzt - nchstes Zeichen prfen }
  end;
  s:=left(s,maxlen);      { String auf 'maxlen' krzen (wegen Headern) }
end;

procedure IBMToIso1(var data; size:word); assembler;
asm
          mov    cx,size
          jcxz   @noconv2
          les    di,data
          mov    bx,offset IBM2ISOtab
          cld
@isolp2:  mov    al,es:[di]
          xlat
          stosb
          loop   @isolp2
@noconv2:
end;

procedure _IBM2ISO(var s:string); assembler;
asm
     push  es
     cld
     mov   bx,offset IBM2ISOtab
     les   si,s
     segES lodsb                     { Stringlnge }
     mov   cl,al
     xor   ch,ch
     jcxz  @@2
@@1: segES lodsb
     xlat
     mov   es:[si-1],al
     loop  @@1
@@2: pop   es
end;


procedure IBM2ISO(var s:string; const maxlen:byte; const multichar:boolean);
begin
  if multichar then
    _IBM2ISOMultiChar(s,maxlen);
  _IBM2ISO(s);
end;


procedure _IBMToIsoMultiChar(var data; var bytes:word); assembler;
asm
     mov  dx,0
     les  di,bytes
     mov  cx,es:[di]
     cmp  cx,0
     je   @ende
     les  di,data
     lea  si,[di+2000]
     cld
@multilp:
     mov  al,es:[si]               { IBM -> Multi-ASCIIs }
     cmp  al,19                    { '' }
     je   @ausrufz
     cmp  al,26                    { Ctrl-Z (SUB) }
     je   @ctrl_z
     cmp  al,27                    { Esc }
     je   @esc
     cmp  al,158                   { '' }
     je   @esp
     cmp  al,159                   { '' }
     je   @nlg
     cmp  al,227                   { '' }
     je   @pi
     cmp  al,232                   { '' }
     je   @cap_phi
     cmp  al,233                   { '' }
     je   @theta
     cmp  al,237                   { '' }
     je   @sml_phi
     cmp  al,239                   { '' }
     je   @intsect
     cmp  al,242                   { '' }
     je   @ge
     cmp  al,243                   { '' }
     je   @le
     cmp  al,252                   { '' }
     je   @hoch_n
     cmp  byte ptr iso15,0         { 0 = false => ISO1 }
     jne  @iso15
     jmp  @noconv
@ausrufz:
     mov  ax,'!!'
     jmp  @conv
@ctrl_z:
     mov  ax,'>-'
     jmp  @conv
@esc:
     mov  ax,'-<'
     jmp  @conv
@esp:
     mov  al,'E'
     stosb
     inc  dx
     mov  ax,'PS'
     jmp  @conv
@nlg:
     mov  al,'N'
     stosb
     inc  dx
     mov  ax,'GL'
     jmp  @conv
@pi:
     mov  ax,'ip'
     jmp  @conv
@cap_phi:
     mov  ax,'hP'
     jmp  @conv
@theta:
     mov  ax,'hT'
     jmp  @conv
@sml_phi:
     mov  ax,'hp'
     jmp  @conv
@intsect:
     mov  ax,'\/'
     jmp  @conv
@ge:
     mov  ax,'=>'
     jmp  @conv
@le:
     mov  ax,'=<'
     jmp  @conv
@hoch_n:
     mov  ax,'n^'
     jmp  @conv
@iso15:
     cmp  al,171
     je   @half
     cmp  al,172
     je   @quarter
     jmp  @noconv
@half:
     mov  al,'1'
     stosb
     inc  dx
     mov  ax,'2/'
     jmp  @conv
@quarter:
     mov  al,'1'
     stosb
     inc  dx
     mov  ax,'4/'
     jmp  @conv
@conv:
     stosw
     inc  dx
     cmp  dx,2000
     jae  @ende                    { Konvertierpuffer voll :-( }
     inc  si
     dec  cx
     jne  @multilp
     jmp  @ende
@noconv:
     stosb
     inc  si
     dec  cx
     jne  @multilp
     jmp  @ende
@ende:
     les  di,bytes
     add  es:[di],dx
end;


procedure _IBMToIso(var data; size:word); assembler;
asm
     mov    cx,size
     jcxz   @noconv2
     les    di,data
     mov    bx,offset IBM2ISOtab
     cld
@isolp2:
     mov    al,es:[di]
     xlat
     stosb
     loop   @isolp2
@noconv2:
end;


procedure IBMToIso(var data; var size:word; const multichar:boolean);
begin
  if multichar then
    _IBMToIsoMultiChar(data,size);  { verndert evtl. size }
  _IBMToIso(data,size);
end;


{ Stringkonvertierung ISO/Win/Mac => IBM }

procedure ISO2IBM(var s:string; const charset:word); assembler;
asm
     push  es
     cld
     mov   ax,charset
     cmp   ax,cs_iso8859_1
     jne   @@iso15
     mov   bx,offset WIN1252_2IBMtab - 128
     jmp   @@conv
@@iso15:
     cmp   ax,cs_iso8859_15
     jne   @@win1252
     mov   bx,offset ISO15_2IBMtab - 128
     jmp   @@conv
@@win1252:
     cmp   ax,cs_win1252
     jne   @@win1250
     mov   bx,offset WIN1252_2IBMtab - 128
     jmp   @@conv
@@win1250:
     cmp   ax,cs_win1250
     jne   @@win1254
     mov   bx,offset WIN1250_2IBMtab - 128
     jmp   @@conv
@@win1254:
     cmp   ax,cs_win1254
     jne   @@win1257
     mov   bx,offset WIN1254_2IBMtab - 128
     jmp   @@conv
@@win1257:
     cmp   ax,cs_win1257
     jne   @@iso2
     mov   bx,offset WIN1257_2IBMtab - 128
     jmp   @@conv
@@iso2:
     cmp   ax,cs_iso8859_2
     jne   @@iso3
     mov   bx,offset ISO2_2IBMtab - 128
     jmp   @@conv
@@iso3:
     cmp   ax,cs_iso8859_3
     jne   @@iso4
     mov   bx,offset ISO3_2IBMtab - 128
     jmp   @@conv
@@iso4:
     cmp   ax,cs_iso8859_4
     jne   @@iso9
     mov   bx,offset ISO4_2IBMtab - 128
     jmp   @@conv
@@iso9:
     cmp   ax,cs_iso8859_9
     jne   @@iso10
     mov   bx,offset WIN1254_2IBMtab - 128
     jmp   @@conv
@@iso10:
     cmp   ax,cs_iso8859_10
     jne   @@iso13
     mov   bx,offset ISO10_2IBMtab - 128
     jmp   @@conv
@@iso13:
     cmp   ax,cs_iso8859_13
     jne   @@iso14
     mov   bx,offset ISO13_2IBMtab - 128
     jmp   @@conv
@@iso14:
     cmp   ax,cs_iso8859_14
     jne   @@iso16
     mov   bx,offset ISO14_2IBMtab - 128
     jmp   @@conv
@@iso16:
     cmp   ax,cs_iso8859_16
     jne   @@cp850
     mov   bx,offset ISO16_2IBMtab - 128
     jmp   @@conv
@@cp850:
     cmp   ax,cs_cp_850
     jne   @@cp858
     mov   bx,offset CP850_2IBMtab - 128
     jmp   @@conv
@@cp858:
     cmp   ax,cs_cp_858
     jne   @@noconv
     mov   bx,offset CP858_2IBMtab - 128
@@conv:
     les   si,s
     segES lodsb                     { Stringlnge }
     mov   cl,al
     xor   ch,ch
     jcxz  @@2
@@1: segES lodsb
     cmp   al,127
     jbe   @@3
     xlat
     mov   es:[si-1],al
@@3: loop  @@1
@@noconv:
@@2: pop   es
end;


{ Blockkonvertierung ISO/Win/Mac => IBM }

procedure IsoToIBM(var data; size:word; const charset:word); assembler;
asm
     mov   cx,size
     cmp   cx,0
     je    @noconv
     mov   ax,charset
     cmp   ax,cs_iso8859_1
     jne   @iso15
     mov   bx,offset WIN1252_2IBMtab - 128
     jmp   @conv
@iso15:
     cmp   ax,cs_iso8859_15
     jne   @win1252
     mov   bx,offset ISO15_2IBMtab - 128
     jmp   @conv
@win1252:
     cmp   ax,cs_win1252
     jne   @win1250
     mov   bx,offset WIN1252_2IBMtab - 128
     jmp   @conv
@win1250:
     cmp   ax,cs_win1250
     jne   @win1254
     mov   bx,offset WIN1250_2IBMtab - 128
     jmp   @conv
@win1254:
     cmp   ax,cs_win1254
     jne   @win1257
     mov   bx,offset WIN1254_2IBMtab - 128
     jmp   @conv
@win1257:
     cmp   ax,cs_win1257
     jne   @iso2
     mov   bx,offset WIN1257_2IBMtab - 128
     jmp   @conv
@iso2:
     cmp   ax,cs_iso8859_2
     jne   @iso3
     mov   bx,offset ISO2_2IBMtab - 128
     jmp   @conv
@iso3:
     cmp   ax,cs_iso8859_3
     jne   @iso4
     mov   bx,offset ISO3_2IBMtab - 128
     jmp   @conv
@iso4:
     cmp   ax,cs_iso8859_4
     jne   @iso9
     mov   bx,offset ISO4_2IBMtab - 128
     jmp   @conv
@iso9:
     cmp   ax,cs_iso8859_9
     jne   @iso10
     mov   bx,offset WIN1254_2IBMtab - 128
     jmp   @conv
@iso10:
     cmp   ax,cs_iso8859_10
     jne   @iso13
     mov   bx,offset ISO10_2IBMtab - 128
     jmp   @conv
@iso13:
     cmp   ax,cs_iso8859_13
     jne   @iso14
     mov   bx,offset ISO13_2IBMtab - 128
     jmp   @conv
@iso14:
     cmp   ax,cs_iso8859_14
     jne   @iso16
     mov   bx,offset ISO14_2IBMtab - 128
     jmp   @conv
@iso16:
     cmp   ax,cs_iso8859_16
     jne   @cp850
     mov   bx,offset ISO16_2IBMtab - 128
     jmp   @conv
@cp850:
     cmp   ax,cs_cp_850
     jne   @cp858
     mov   bx,offset CP850_2IBMtab - 128
     jmp   @conv
@cp858:
     cmp   ax,cs_cp_858
     jne   @mac
     mov   bx,offset CP858_2IBMtab - 128
     jmp   @conv
@mac:
     cmp   ax,cs_macroman
     jne   @noconv
     mov   bx,offset Mac2IBMtab - 128
@conv:
     les   di,data
     cld
@convlp:
     mov   al,es:[di]
     or    al,al
     jns   @ii1
     xlat
@ii1:
     stosb
     dec   cx
     jne   @convlp
@noconv:
end;

procedure Iso1ToIBM(var data; size:word); assembler;
asm
          mov    cx,size
          jcxz   @noconv1
          les    di,data
          mov    bx,offset ISO1_2IBMtab - 128
          cld
@isolp1:  mov    al,es:[di]
          or     al,al
          jns    @ii1
          xlat
@ii1:     stosb
          loop   @isolp1
@noconv1:
end;


procedure Mac2IBM(var data; size:word); {&uses ebx, esi} assembler;
asm
          mov    bx,offset Mac2IBMtab - 128
          les    si,data
          mov    cx,size
          jcxz   @xende
          jmp    @xloop
@xloop:   mov    al,es:[si]
          inc    si
          cmp    al,127
          ja     @trans
          loop   @xloop
          jmp    @xende
@trans:   xlat
          mov    es:[si-1],al
          loop   @xloop
@xende:
end;


Function ConvertUnicode(ucs:dword):char;assembler;
Const
  Error     = 177;   { unkonvertierbar }
  Anzahl    = 241;
asm
        mov al,error
        mov bx,word ptr ucs
        mov cx,word ptr ucs+2
        or cx,cx               { > $FFFF }
        jne @end
        cmp bx,$80             { < $0080 }
        jb @end
        cmp bh,0               { < $00FF }
        jne @noiso
        mov al,byte ptr WIN1252_2IBMtab[bx-128]
        jmp @end
@noiso: mov ax,bx
        mov di,offset @unicode
        mov si,di
        push cs
        pop es
        mov cx,Anzahl
        repne scasw
        mov al,error
        jne @end
        sub di,si
        shr di,1
        mov al,byte ptr cs:[@Asciicode+di-1]
        jmp @end

@Unicode:    { CP437 + Euro }
             dw $20ac,$20a7,$0192,$2310,$2591,$2592,$2593,$2502,$2524,$2561
             dw $2562,$2556,$2555,$2563,$2551,$2557,$255d,$255c,$255b,$2510
             dw $2514,$2534,$252c,$251c,$2500,$253c,$255e,$255f,$255a,$2554
             dw $2569,$2566,$2560,$2550,$256c,$2567,$2568,$2564,$2565,$2559
             dw $2558,$2552,$2553,$256b,$256a,$2518,$250c,$2588,$2584,$258c
             dw $2590,$2580,$03b1,$0393,$03c0,$03a3,$03c3,$03c4,$03a6,$0398
             dw $03a9,$03b4,$221e,$03c6,$03b5,$2229,$2261,$2265,$2264,$2320
             dw $2321,$2248,$2219,$221a,$207f,$25a0
             { ISO-8859-2 bis -16, CP850/858 }
             dw $0100,$0101,$0102,$0103,$0104,$0105,$0106,$0107,$0108,$0109
             dw $010a,$010b,$010c,$010d,$010e,$010f,$0110,$0111,$0112,$0113
             dw $0116,$0117,$0118,$0119,$011a,$011b,$011c,$011d,$011e,$011f
             dw $0120,$0121,$0122,$0123,$0124,$0125,$0126,$0127,$0128,$0129
             dw $012a,$012b,$012e,$012f,$0130,$0131,$0134,$0135,$0136,$0137
             dw $0138,$0139,$013a,$013b,$013c,$013d,$013e,$0141,$0142,$0143
             dw $0144,$0145,$0146,$0147,$0148,$014a,$014b,$014c,$014d,$0150
             dw $0151,$0152,$0153,$0154,$0155,$0156,$0157,$0158,$0159,$015a
             dw $015b,$015c,$015d,$015e,$015f,$0160,$0161,$0162,$0163,$0164
             dw $0165,$0166,$0167,$0168,$0169,$016a,$016b,$016c,$016d,$016e
             dw $016f,$0170,$0171,$0172,$0173,$0174,$0175,$0176,$0177,$0178
             dw $0179,$017a,$017b,$017c,$017d,$017e,$0218,$0219,$021a,$021b
             dw $02c6,$02dc,$02dd,$1e02,$1e03,$1e0a,$1e0b,$1e1e,$1e1f,$1e40
             dw $1e41,$1e56,$1e57,$1e60,$1e61,$1e6a,$1e6b,$1e80,$1e81,$1e82
             dw $1e83,$1e84,$1e85,$1ef2,$1ef3,$2013,$2014,$2015,$2017,$2018
             dw $2019,$201a,$201c,$201d,$201e,$2022,$2039,$203a
             { MacRoman }
             dw $02da,$2044,$220f,$2211,$222b,$fb01,$fb02

@Asciicode:  db  euro, $9e , $9f , $a9 , $b0 , $b1 , $b2 , $b3 , $b4 , $b5
             db  $b6 , $b7 , $b8 , $b9 , $ba , $bb , $bc , $bd , $be , $bf
             db  $c0 , $c1 , $c2 , $c3 , $c4 , $c5 , $c6 , $c7 , $c8 , $c9
             db  $ca , $cb , $cc , $cd , $ce , $cf , $d0 , $d1 , $d2 , $d3
             db  $d4 , $d5 , $d6 , $d7 , $d8 , $d9 , $da , $db , $dc , $dd
             db  $de , $df , $e0 , $e2 , $e3 , $e4 , $e5 , $e7 , $e8 , $e9
             db  $ea , $eb , $ec , $ed , $65 , $ef , $f0 , $f2 , $f3 , $f4
             db  $f5 , $f7 , $f9 , $fb , $fc , $fe
             db  $41 , $61 , $41 , $61 , $41 , $61 , $43 , $63 , $43 , $63
             db  $43 , $63 , $43 , $63 , $44 , $64 , $44 , $64 , $45 , $65
             db  $45 , $65 , $45 , $65 , $45 , $65 , $47 , $67 , $47 , $67
             db  $47 , $67 , $47 , $67 , $48 , $68 , $48 , $68 , $49 , $69
             db  $49 , $69 , $49 , $69 , $49 , $69 , $4a , $6a , $4b , $6b
             db  $6b , $4c , $6c , $4c , $6c , $4c , $6c , $4c , $6c , $4e
             db  $6e , $4e , $6e , $4e , $6e , $4e , $6e , $4f , $6f , $99
             db  $94 , $4f , $6f , $52 , $72 , $52 , $72 , $52 , $72 , $53
             db  $73 , $53 , $73 , $53 , $73 , $53 , $73 , $54 , $74 , $54
             db  $74 , $54 , $74 , $55 , $75 , $55 , $75 , $55 , $75 , $55
             db  $75 , $9a , $81 , $55 , $75 , $57 , $77 , $59 , $79 , $59
             db  $5a , $7a , $5a , $7a , $5a , $7a , $53 , $73 , $54 , $74
             db  $5e , $7e , $22 , $42 , $62 , $44 , $64 , $46 , $66 , $4d
             db  $6d , $50 , $70 , $53 , $73 , $54 , $74 , $57 , $77 , $57
             db  $77 , $57 , $77 , $59 , $79 , $c4 , $c4 , $c4 , $5f , $27
             db  $27 , $2c , $22 , $22 , $22 , $2a , $3c , $3e
             db  $f8 , $2f , $e3 , $e4 , $f4 , $66 , $66
@end:
end;


{ my: UTF-8-Routine erweitert und gefixt fr beliebig lange Strings, die }
{     aus max. 250 Zeichen langen Teilstrings bestehen drfen.   04/2004 }

procedure UTF8ToIBM(var s:string);
const sc_rest : string[3]='';
            k : integer=0;
var       i,j : integer;
           sc : record case integer of
                  0 : (s:string[4]);
                  1 : (b:array[0..4] of byte);
                end;
          ucs : longint;

  { my: Prfung auf gltiges UTF-8 gem "The Unicode Standard, Version 4.0" }
  {     (siehe <http://www.unicode.org/versions/Unicode4.0.0/ch03.pdf>, 3.9) }
  {     und RFC3629 (November 2003, supersedes RFC2279)                      }

  function valid_UTF8(const cvt:string):boolean;
  var ok : boolean;
      sp : byte;
  begin
    ok:=byte(cvt[1]) in [$c2..$f4];
    sp:=1;
    while ok and (sp<length(cvt)) do
    begin
      inc(sp);
      if sp=2 then                               { 2. Octet prfen!         }
      begin
        case byte(cvt[1]) of
          $e0 : ok:=byte(cvt[sp]) in [$a0..$bf]; { 3-Octet-Sequenzen mit E0 }
          $ed : ok:=byte(cvt[sp]) in [$80..$9f]; { 3-Octet-Sequenzen mit ED }
          $f0 : ok:=byte(cvt[sp]) in [$90..$bf]; { 4-Octet-Sequenzen mit F0 }
          $f4 : ok:=byte(cvt[sp]) in [$80..$8f]; { 4-Octet-Sequenzen mit F4 }
        else
          ok:=byte(cvt[sp]) in [$80..$bf];       { alle brigen Sequenzen   }
        end;
      end
      else ok:=byte(cvt[sp]) in [$80..$bf];      { 3.-4. Octet prfen       }
    end;
    valid_UTF8:=ok;
  end;

  procedure Get_UCS;
  var jj : integer;
  begin
    for jj:=0 to k-1 do sc.b[1]:=sc.b[1] and not ($80 shr jj);
    for jj:=2 to k do sc.b[jj]:=sc.b[jj] and $3f;
    ucs:=0;
    for jj:=0 to k-1 do ucs:=ucs or (longint(sc.b[k-jj]) shl (jj*6));
  end;

  procedure CharToIBM(var c:char);
  var b : string[1];
  begin
    if byte(c)<$80 then exit;
    b:=c;
    CharsetToIBM('',b);
    c:=b[1];
  end;

begin
  i:=0;
  if not start_of_UTF and (sc_rest<>'') then     { fortgesetzte Decodierung }
  begin
    k:=k-length(sc_rest);
    if (k>0) and (k<=length(s)) and
       valid_UTF8(sc_rest+left(s,k)) then          { gltige UTF-8-Sequenz? }
    begin
      sc.s:=sc_rest+left(s,k);      { zerrissene Sequenz separat decodieren }
      s:=mid(s,k);
      i:=1;                              { Position im String aktualisieren }
      k:=length(sc.s);
      Get_UCS;
      s[1]:=ConvertUnicode(ucs);
    end
    else begin
      CharsetToIBM('',sc_rest);    { ungltiges UTF-8 (falsch deklariert)?! } 
      s:=sc_rest+s;
      i:=length(sc_rest);                { Position im String aktualisieren }
    end;
  end;
  sc_rest:='';
  k:=0;
  while i<length(s) do
  begin
    inc(i);
    if valid_UTF8(s[i]) then           { gltiges 1. Byte in UTF-8-Sequenz? }
    begin
      k:=0;
      for j:=0 to 7 do
        if byte(s[i]) and ($80 shr j)=($80 shr j) then inc(k) else break;
      sc.s:=copy(s,i,k);
      if valid_UTF8(sc.s) then                     { gltige UTF-8-Sequenz? }
      begin
        if length(sc.s)=k then
        begin
          delete(s,i,k-1);
          Get_UCS;
          s[i]:=ConvertUnicode(ucs);
        end
        else if end_of_UTF then      { es kommen keine neuen UTF-Daten mehr }
          CharToIBM(s[i])
        else begin                      { unvollstndige Sequenz => sc_rest }
          sc_rest:=sc.s;
          delete(s,i,length(sc.s));
          break;
        end;
      end
      else CharToIBM(s[i]);        { ungltiges UTF-8 (falsch deklariert)?! }
    end
    else CharToIBM(s[i]);          { ungltiges UTF-8 (falsch deklariert)?! }
  end;
end;


{ my: UTF-7-Routine erweitert und gefixt fr beliebig lange Strings, die }
{     aus max. 248 Zeichen langen Teilstrings bestehen drfen.   04/2004 }

procedure UTF7ToIBM(var s:string);
const b64alphabet='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
      sc_rest  : string[4]='';   { Reststring von UTF-7-Sequenz  }
      b64_rest : string[1]='';   { Restbyte nach b64-Decodierung }
var       i,j  : integer;
           s1  : string;
          ucs  : smallword;
    continued,
    incomplete : boolean;

begin
  continued:=false;
  incomplete:=false;
  if not start_of_UTF then                 { evtl. fortgesetzte Decodierung }
  begin
    if (sc_rest='*') and                 { einzelnes '+' aus fortgesetzter, }
       (((s<>'') and                     { unvollstndiger Sequenz, auf     }
         (cpos(s[1],b64alphabet)=0)) or  { gltiges 1. Byte prfen und '+'  }
        (s='')) then                     { (aka '*') ggf. verwerfen         }
    begin
      sc_rest:='';
      if firstchar(s)='-' then delfirst(s);   { Ende der vorherigen Sequenz }
    end;
    if firstchar(sc_rest)='*' then
    begin
      sc_rest[1]:='+';
      continued:=true;
    end;
    s:=sc_rest+s;
  end
  else b64_rest:='';
  sc_rest:='';
  CharsetToIBM('',s);           { fr Nachrichten, die als UTF-7 deklariert }
  i:=1;                         { sind, aber 8bit-Zeichen enthalten         }
  j:=posn('+',s,i);
  while j<>0 do
  begin
    i:=j;
    inc(j);
    while (j<=length(s)) and (cpos(s[j],b64alphabet)<>0) do inc(j);
    if (j<=length(s)) or end_of_UTF then             { vollstndige Sequenz }
    begin
      if not continued and               { Sequenz < 3 Zeichen => ungltig! }
         (j>i+1) and (j-i<4) then
        j:=i+1
      else if (j<=length(s)) and (s[j]='-') then
        inc(j);
      if j>i+1 then                               { einzelnes '+' abfangen! }
      begin
        s1:=copy(s,i,j-i);                  { zu decodierende Sequenz => s1 }
        delete(s,i,j-i);
      end
      else s1:='';
    end
    else begin   { j>length(s) and not end_of_UTF => unvollstndige Sequenz }
      incomplete:=true;
      if j-i<=4 then      { unvollstndige Sequenz bis 3 Zeichen => sc_rest }
      begin               { (4 Bytes bilden eine abgeschlossene Einheit)    }
        sc_rest:=right(s,j-i);                                 { inkl. '+'! }
        s:=left(s,i-1);
        break;
      end
      else begin                      { unvollstndige Sequenz > 3 Zeichen, }
        j:=(j-i-1) mod 4;             { Rest nach Div. durch 4 => sc_rest   }
        if j>0 then
        begin
          sc_rest:=right(s,j);
          dec(byte(s[0]),j);
        end;
        sc_rest:='*'+sc_rest;   { '*' (= '+' bei fortgesetzter Decodierung) }
        j:=byte(s[0])+1;        { auch bei leerem sc_rest bertragen!       }
      end;
      s1:=right(s,j-i);                 { zu decodierende Teilsequenz => s1 }
      s:=left(s,i-1);
    end;
    if s1<>'' then             { einzelnes '+' bei not incomplete abfangen! }
    begin
      if s1='+-' then
        s1:='+'
      else begin
        if s1[1]='+' then delfirst(s1);
        if not incomplete then
        begin
          if lastchar(s1)='-' then dellast(s1);
          while (length(s1) mod 4<>0) do s1:=s1+'=';
        end;
        DecodeBase64(s1);
        if b64_rest<>'' then
        begin
          s1:=b64_rest+s1;
          b64_rest:='';
        end;
        if odd(byte(s1[0])) then
        begin
          if incomplete then
            b64_rest:=lastchar(s1);        { berhngendes Byte => b64_rest }
          dec(byte(s1[0]));
        end;
        j:=1;
        while length(s1)>j do
        begin
          ucs:=word(s1[j]) shl 8+word(s1[j+1]);
          if (ucs<$80)
            then s1[j]:=char(ucs)
            else s1[j]:=convertUnicode(ucs);
          inc(j);
          delete(s1,j,1);
        end;
      end;
      insert(s1,s,i);
    end;
    j:=posn('+',s,i+max(length(s1),1));
  end;
end;


{ IANA-Charset-Alias aller eingehend }
{ untersttzten Charsets zurckgeben }

function RFC_CharsetName(const n:word):string;
begin
  Case n of
    0              : RFC_CharsetName:='US-ASCII';
    cs_cp_437      : RFC_CharsetName:='IBM437';
    cs_cp_850      : RFC_CharsetName:='IBM850';
    cs_cp_858      : RFC_CharsetName:='IBM00858';
    cs_iso8859_1   : RFC_CharsetName:='ISO-8859-1';
    cs_iso8859_2   : RFC_CharsetName:='ISO-8859-2';
    cs_iso8859_3   : RFC_CharsetName:='ISO-8859-3';
    cs_iso8859_4   : RFC_CharsetName:='ISO-8859-4';
    cs_iso8859_9   : RFC_CharsetName:='ISO-8859-9';
    cs_iso8859_10  : RFC_CharsetName:='ISO-8859-10';
    cs_iso8859_13  : RFC_CharsetName:='ISO-8859-13';
    cs_iso8859_14  : RFC_CharsetName:='ISO-8859-14';
    cs_iso8859_15  : RFC_CharsetName:='ISO-8859-15';
    cs_iso8859_16  : RFC_CharsetName:='ISO-8859-16';
    cs_win1250     : RFC_CharsetName:='windows-1250';
    cs_win1252     : RFC_CharsetName:='windows-1252';
    cs_win1254     : RFC_CharsetName:='windows-1254';
    cs_win1257     : RFC_CharsetName:='windows-1257';
    cs_macroman    : RFC_CharsetName:='macintosh';
    cs_utf_7       : RFC_CharsetName:='UTF-7';
    cs_utf_8       : RFC_CharsetName:='UTF-8';
  end;
  iso15:=n=cs_iso8859_15;
end;


{ ZConnect-Charset-Alias aller ISO-8859-x-Charsets zurckgeben    }
{ (falls anderer Charset, mglichst IANA-Charset-Alias verwenden) }

function ZC_CharsetName(s:string):string;
begin
  s:=any2rfc_charset(s);
  if (lstr(left(s,9))='iso-8859-') and (byte(s[0])=10) and
     (ival(lastchar(s)) in [1..9]) then
    s:='ISO'+lastchar(s);
  ZC_CharsetName:=s;
end;


{ US-ASCII / CP437 - diese Charsets werden nicht konvertiert }

function ascii_charset(const s:string):boolean;
begin
  ascii_charset:=pos(';'+s+';',';us-ascii;ascii;csascii;iso-ir-6;'  { IANA  }
                              + 'ansi_x3.4-1968;ansi_x3.4-1986;'
                              + 'iso_646.irv:1991;'
                              + 'iso646-us;us;ibm367;cp367;') > 0;
end;

function ibm_charset(const s:string):boolean;
begin
  ibm_charset:=pos(';'+s+';',';ibm437;cp437;437;cspc8codepage437;'  { IANA  }
                              + 'osf100201b5;') > 0;                { iconv }
end;

function local_charset(const s:string):boolean;
begin
  local_charset:=ascii_charset(s) or ibm_charset(s);
end;


{ ISO1    / ISO2    / ISO3  / ISO4  / ISO9    / ISO10   }
{ ISO13   / ISO14   / ISO15 / ISO16 / Win1250 / Win1252 }
{ Win1254 / Win1257 / CP850 / CP858 / MacRoman          }
{ diese Charsets drfen und mssen konvertiert werden   }

function convertible_charset(const s:string):word;

  function iso_8859_1:boolean;
  begin
    iso_8859_1:=pos(';'+s+';',';iso-8859-1;iso_8859-1;latin1;'      { IANA  }
                             + 'l1;iso_8859-1:1987;iso-ir-100;'
                             + 'csisolatin1;ibm819;cp819;'
                             + 'iso8859-1;iso88591;8859_1;'         { iconv }
                             + 'osf00010001;'
                             + 'iso1;') > 0;                     { ZConnect }
  end;

  function iso_8859_2:boolean;
  begin
    iso_8859_2:=pos(';'+s+';',';iso-8859-2;iso_8859-2;latin2;'      { IANA  }
                             + 'l2;iso_8859-2:1987;iso-ir-101;'
                             + 'csisolatin2;'
                             + 'iso8859-2;iso88592;8859_2;'         { iconv }
                             + 'osf00010002;ibm912;cp912;'
                             + 'iso2;') > 0;                     { ZConnect }
  end;

  function iso_8859_3:boolean;
  begin
    iso_8859_3:=pos(';'+s+';',';iso-8859-3;iso_8859-3;latin3;'      { IANA  }
                             + 'l3;iso_8859-3:1988;iso-ir-109;'
                             + 'csisolatin3;'
                             + 'iso8859-3;iso88593;8859_3;'         { iconv }
                             + 'osf00010003;'
                             + 'iso3;') > 0;                     { ZConnect }
  end;

  function iso_8859_4:boolean;
  begin
    iso_8859_4:=pos(';'+s+';',';iso-8859-4;iso_8859-4;latin4;'      { IANA  }
                             + 'l4;iso_8859-4:1988;iso-ir-110;'
                             + 'csisolatin4;'
                             + 'iso8859-4;iso88594;8859_4;'         { iconv }
                             + 'osf00010004;'
                             + 'iso4;') > 0;                     { ZConnect }
  end;

  function iso_8859_9:boolean;
  begin
    iso_8859_9:=pos(';'+s+';',';iso-8859-9;iso_8859-9;latin5;'      { IANA  }
                             + 'l5;iso_8859-9:1989;iso-ir-148;'
                             + 'csisolatin5;'
                             + 'iso8859-9;iso88599;8859_9;'         { iconv }
                             + 'osf00010009;'
                             + 'ibm920;cp920;ts-5881;ecma-128;'
                             + 'iso9;') > 0;                     { ZConnect }
  end;

  function iso_8859_10:boolean;
  begin
    iso_8859_10:=pos(';'+s+';',';iso-8859-10;iso_8859-10;latin6;'   { IANA  }
                              + 'l6;iso_8859-10:1992;iso-ir-157;'
                              + 'csisolatin6;'
                              + 'iso8859-10;iso885910;'             { iconv }
                              + 'osf0001000a;'
                              + 'iso10;') > 0;                   { ZConnect }
  end;

  function iso_8859_13:boolean;
  begin
    iso_8859_13:=pos(';'+s+';',';iso-8859-13;'                      { IANA  }
                              + 'iso_8859-13;iso8859-13;'           { iconv }
                              + 'iso_885913;'
                              + 'iso-ir-179;latin7;l7;baltic;'
                              + 'iso13;') > 0;                   { ZConnect }
  end;

  function iso_8859_14:boolean;
  begin
    iso_8859_14:=pos(';'+s+';',';iso-8859-14;iso_8859-14;latin8;'   { IANA  }
                              + 'l8;iso_8859-14:1998;iso-ir-199;'
                              + 'iso-celtic;'
                              + 'iso8859-14;iso885914;'             { iconv }
                              + 'iso14;') > 0;                   { ZConnect }
  end;

  function iso_8859_15:boolean;
  begin
    iso_8859_15:=pos(';'+s+';',';iso-8859-15;iso_8859-15;latin-9;'  { IANA  }
                              + 'iso_8859-15:1998;iso8859-15;'      { iconv }
                              + 'iso885915;'
                              + 'iso-ir-203;'
                              + 'iso15;') > 0;                   { ZConnect }
  end;

  function iso_8859_16:boolean;
  begin
    iso_8859_16:=pos(';'+s+';',';iso-8859-16;iso_8859-16;latin10;'  { IANA  }
                              + 'l10;iso_8859-16:2001;iso-ir-226;'
                              + 'iso8859-16;iso885916;'             { iconv }
                              + 'iso16;') > 0;                   { ZConnect }
  end;

  function win_1250:boolean;
  begin
    win_1250:=pos(';'+s+';',';windows-1250;'                        { IANA  }
                           + 'cp1250;ms-ee;') > 0;                  { iconv }
  end;

  function win_1252:boolean;
  begin
    win_1252:=pos(';'+s+';',';windows-1252;'                        { IANA  }
                           + 'cp1252;ms-ansi;') > 0;                { iconv }
  end;

  function win_1254:boolean;
  begin
    win_1254:=pos(';'+s+';',';windows-1254;'                        { IANA  }
                           + 'cp1254;ms-turk;') > 0;                { iconv }
  end;

  function win_1257:boolean;
  begin
    win_1257:=pos(';'+s+';',';windows-1257;'                        { IANA  }
                           + 'cp1257;winbaltrim;') > 0;             { iconv }
  end;

  function mac_roman:boolean;
  begin
    mac_roman:=pos(';'+s+';',';macintosh;mac;csmacintosh;'          { IANA  }
                            + 'macroman;'                           { iconv }
                            + 'x-mac-roman;') > 0;      { Netscape/Mozilla? }
  end;

  function cp_850:boolean;
  begin
    cp_850:=pos(';'+s+';',';ibm850;cp850;850;cspc850multilingual;'  { IANA  }
                         + 'osf10020352;') > 0;                     { iconv }
  end;

  function cp_858:boolean;
  begin
    cp_858:=pos(';'+s+';',';ibm00858;ccsid00858;cp00858;'           { IANA  }
                         + 'pc-multilingual-850+euro;') > 0;
  end;

  function utf_7:boolean;
  begin
    utf_7:=pos(';'+s+';',';utf-7;utf7;') > 0;
  end;

  function utf_8:boolean;
  begin
    utf_8:=pos(';'+s+';',';utf-8;utf8;') > 0;
  end;

begin
  { Einzelfunktionen aufrufen wegen Stackgre! }
  if iso_8859_1  then convertible_charset:=cs_iso8859_1  else
  if iso_8859_15 then convertible_charset:=cs_iso8859_15 else
  if win_1252    then convertible_charset:=cs_win1252    else
  if win_1250    then convertible_charset:=cs_win1250    else
  if win_1254    then convertible_charset:=cs_win1254    else
  if win_1257    then convertible_charset:=cs_win1257    else
  if utf_8       then convertible_charset:=cs_utf_8      else
  if iso_8859_2  then convertible_charset:=cs_iso8859_2  else
  if iso_8859_3  then convertible_charset:=cs_iso8859_3  else
  if iso_8859_4  then convertible_charset:=cs_iso8859_4  else
  if iso_8859_9  then convertible_charset:=cs_iso8859_9  else
  if iso_8859_10 then convertible_charset:=cs_iso8859_10 else
  if iso_8859_13 then convertible_charset:=cs_iso8859_13 else
  if iso_8859_14 then convertible_charset:=cs_iso8859_14 else
  if iso_8859_16 then convertible_charset:=cs_iso8859_16 else
  if utf_7       then convertible_charset:=cs_utf_7      else
  if cp_850      then convertible_charset:=cs_cp_850     else
  if cp_858      then convertible_charset:=cs_cp_858     else
  if mac_roman   then convertible_charset:=cs_macroman   else
  convertible_charset:=0;
end;

{ ISO1 / ISO2 / ISO9 / ISO15 / Win1252 / CP850 / CP858 }
{ diese Charsets drfen und mssen konvertiert werden  }
function iso_charset(s:string):word;
begin
  iso_charset:=0;
  if pos(';'+s+';',';iso-8859-1;iso_8859-1;iso_8859-1:1987;iso-ir-100;latin1;l1;csisolatin1;ibm819;cp819;') > 0 then
    iso_charset:=cs_iso8859_1 else
  if pos(';'+s+';',';iso-8859-2;iso_8859-2;iso_8859-2:1987;iso-ir-101;latin2;l2;csisolatin2;') > 0 then
    iso_charset:=cs_iso8859_2 else
  if pos(';'+s+';',';iso-8859-9;iso_8859-9;iso_8859-9:1989;iso-ir-148;latin5;l5;csisolatin5;') > 0 then
    iso_charset:=cs_iso8859_9 else
  if pos(';'+s+';',';iso-8859-15;iso_8859-15;latin-9;') > 0 then
    iso_charset:=cs_iso8859_15 else
  if s='windows-1252' then
    iso_charset:=cs_win1252 else
  if pos(';'+s+';',';ibm850;cp850;850;cspc850multilingual;') > 0 then
    iso_charset:=cs_cp_850 else
  if pos(';'+s+';',';ibm00858;ccsid00858;cp00858;pc-multilingual-850+euro;') > 0 then
    iso_charset:=cs_cp_858;
end;


function supported_charset(const s:string):boolean;
begin
  supported_charset:=(s='') or local_charset(s) or (convertible_charset(s)>0);
end;


{ beliebige Charset-Bezeichnung nach RFC/IANA-Identifier konvertieren }

function any2rfc_charset(const charset:string):string;
var cs : word;
     s : string[30];

  { Zeichenstze behandeln, die im Header "CHARSET:" vorkommen knnen    }
  { (manche Gate- und andere Software schert sich offenbar nicht um      }
  { ZC-Standards), nicht aber in 'convertible_charset' behandelt werden  }
  { und deren Bezeichner daher nicht ber 'RFC_CharsetName' gesetzt      }
  { werden kann (speziell fr "uuz -zu" bei vorhandenem Header           }
  { "CHARSET: <charset>")                                                }

  function other2rfc_charset(s:string):string;

    function iso_8859_5:boolean;
    begin
      iso_8859_5:=pos(';'+s+';',';iso-8859-5;iso_8859-5;cyrillic;'  { IANA  }
                               + 'iso_8859-5:1988;iso-ir-144;'
                               + 'csisolatincyrillic;'
                               + 'iso8859-5;iso88595;8859_5;'       { iconv }
                               + 'osf00010005;ibm915;cp915;'
                               + 'iso5;') > 0;                   { ZConnect }
    end;

    function iso_8859_6:boolean;
    begin
      iso_8859_6:=pos(';'+s+';',';iso-8859-6;iso_8859-6;arabic;'    { IANA  }
                               + 'iso_8859-6:1987;iso-ir-127;'
                               + 'ecma-114;asmo-708;'
                               + 'csisolatinarabic;'
                               + 'iso8859-6;iso88596;8859_6;'       { iconv }
                               + 'osf00010006;'
                               + 'ibm1089;cp1089;'
                               + 'iso6;') > 0;                   { ZConnect }
    end;

    function iso_8859_7:boolean;
    begin
      iso_8859_7:=pos(';'+s+';',';iso-8859-7;iso_8859-7;greek;'     { IANA  }
                               + 'greek8;iso_8859-7:1987;'
                               + 'iso-ir-126;ecma-118;elot_928;'
                               + 'csisolatingreek;'
                               + 'iso8859-7;iso88597;8859_7;'       { iconv }
                               + 'osf00010007;'
                               + 'ibm813;cp813;'
                               + 'iso7;') > 0;                   { ZConnect }
    end;

    function iso_8859_8:boolean;
    begin
      iso_8859_8:=pos(';'+s+';',';iso-8859-8;iso_8859-8;hebrew;'    { IANA  }
                               + 'iso_8859-8:1988;iso-ir-138;'
                               + 'csisolatinhebrew;'
                               + 'iso8859-8;iso88598;8859_8;'       { iconv }
                               + 'osf00010008;'
                               + 'ibm916;cp916;'
                               + 'iso8;') > 0;                   { ZConnect }
    end;

    function iso_8859_11:boolean;
    begin
      iso_8859_11:=pos(';'+s+';',';iso-8859-11;iso8859-11;'         { iconv }
                                + 'iso885911;'
                                + 'iso11;') > 0;                 { ZConnect }
    end;

    { *** bei Bedarf ggf. noch um weitere Charsets erweitern *** }

  begin
    { Einzelfunktionen aufrufen wegen Stackgre! }
    if iso_8859_5  then other2rfc_charset:='ISO-8859-5'  else
    if iso_8859_6  then other2rfc_charset:='ISO-8859-6'  else
    if iso_8859_7  then other2rfc_charset:='ISO-8859-7'  else
    if iso_8859_8  then other2rfc_charset:='ISO-8859-8'  else
    if iso_8859_11 then other2rfc_charset:='ISO-8859-11' else
    other2rfc_charset:='';
  end;

begin
  s:=lstr(charset);
  if s<>'' then
  begin
    if ascii_charset(s) then          { US-ASCII }
      s:=RFC_CharsetName(cs_us_ascii)
    else if ibm_charset(s) then       { CP437 }
      s:=RFC_CharsetName(cs_cp_437)
    else begin
      cs:=convertible_charset(s);     { konvertierbarer Charset }
      if cs>0 then
        s:=RFC_CharsetName(cs)
      else begin
        s:=other2rfc_charset(s);      { nicht konvertierbarer Charset }
        if s='' then s:=charset;      { unbekannter Charset :-( }
      end;
    end;
  end;
  any2rfc_charset:=s;
end;


{ Adquaten Windows-Charset zurckgeben, wenn als ISO-8859-x }
{ deklarierte Nachricht Zeichen im Bereich 0x80-0x9F enthlt }
{                                                            }
{ (Hier nie von oder auf Multibyte-Zeichenstze wie UTF-7/8  }
{ zurckfallen, weil die Konvertierroutine in 'DecodeEW' bei }
{ Fallback-Zeichenstzen nicht davon ausgeht, da sich die   }
{ Lnge des konvertierten Strings ndern kann!)              }

function fallback_charset(const n:word):string;
begin
  case n of
    cs_iso8859_1  : fallback_charset:=RFC_CharsetName(cs_win1252);
    cs_iso8859_2  : fallback_charset:=RFC_CharsetName(cs_win1250);
    cs_iso8859_3  : fallback_charset:=RFC_CharsetName(cs_win1254);
    cs_iso8859_4  : fallback_charset:=RFC_CharsetName(cs_win1257);
    cs_iso8859_9  : fallback_charset:=RFC_CharsetName(cs_win1254);
    cs_iso8859_10 : fallback_charset:=RFC_CharsetName(cs_win1257);
    cs_iso8859_13 : fallback_charset:=RFC_CharsetName(cs_win1257);
    cs_iso8859_14 : fallback_charset:=RFC_CharsetName(cs_win1252);
    cs_iso8859_15 : fallback_charset:=RFC_CharsetName(cs_win1252);
    cs_iso8859_16 : fallback_charset:=RFC_CharsetName(cs_win1250);
  else fallback_charset:='';
  end;
end;


procedure CharsetToIBM(charset:string; var s:string);
var cs : word;
begin
  lostring(charset);
  cs:=convertible_charset(charset);
  if cs=cs_utf_7 then UTF7ToIBM(s) else
  if cs=cs_utf_8 then UTF8ToIBM(s) else
  if cs>0 then ISO2IBM(s,cs) else
  if charset='' then ISO2IBM(s,cs_win1252);  { OjE-Fix! }
end;


procedure DecodeBase64(var s:string);
const
  b64tab : array[0..127] of byte =
           ( 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,63, 0, 0, 0,64,
            53,54,55,56,57,58,59,60,61,62, 0, 0, 0, 0, 0, 0,
             0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15,
            16,17,18,19,20,21,22,23,24,25,26, 0, 0, 0, 0, 0,
             0,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,
            42,43,44,45,46,47,48,49,50,51,52, 0, 0, 0, 0, 0);
  b64alphabet='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';
var b1,b2,b3,b4 : byte;
    p1,p2,pad   : byte;
    i: integer;

  function nextbyte:byte;
  var p : byte;
  begin
    nextbyte:=0;
    if p1>length(s) then exit;
    repeat
      if s[p1]>#127 then p:=0
      else p:=b64tab[byte(s[p1])];
      inc(p1);
    until (p>0) or (p1>length(s));
    if p>0 then dec(p);
    nextbyte:=p;
  end;

begin
  if length(s)<4 then s:=''
  else begin
    for i:=1 to length(s) do if cpos(s[i],b64alphabet)=0 then exit;
    if s[length(s)]='=' then begin
      if s[length(s)-1]='=' then pad:=2
      else pad:=1;
      if length(s) mod 4<>0 then pad:=3;
    end
    else pad:=0;
    p1:=1; p2:=1;
    while p1<=length(s) do begin
      b1:=nextbyte; b2:=nextbyte; b3:=nextbyte; b4:=nextbyte;
      s[p2]:=chr(b1 shl 2 + b2 shr 4);
      s[p2+1]:=chr((b2 and 15) shl 4 + b3 shr 2);
      s[p2+2]:=chr((b3 and 3) shl 6 + b4);
      inc(p2,3);
    end;
    s[0]:=chr(p2-1-pad);
  end;
end;


{ my: MIME-quoted-printable => 8bit             }
{     (gibt 'true' zurck, wenn soft line break }
function UnQuotePrintable(var s:string; const eol:byte):boolean;
var p,b : byte;
begin
  UnQuotePrintable:=false;
  if eol>0 then s:=rtrim(s);
  if lastchar(s)='=' then     { soft line break }
  begin
    dellast(s);
    UnQuotePrintable:=true;
  end;
  p:=cpos('=',s);
  if p>0 then while p<length(s)-1 do
  begin
    inc(p);
    if (upcase(s[p]) in ['0'..'9','A'..'F']) and
       (upcase(s[p+1]) in ['0'..'9','A'..'F']) then
    begin
      b:=hexval(ustr(copy(s,p,2)));
      if b>=0 then
      begin
        s[p-1]:=chr(b);
        delete(s,p,2);
      end;
    end;
    while (p<length(s)) and (s[p]<>'=') do inc(p);
  end;
end;

{ ------------------------------------------------------------------------ }
{ my: 'GetHeader' - Neue Routine 08/2002-04/2003                           }
{                                                                          }
{      Lange Headerzeilen gem RFC2822/2047 parsen, ggf. MIME-decodieren  }
{      sowie TABs durch Leerzeichen und Mehrfach-Leerzeichen durch eines   }
{      ersetzen.                                                           }
{ ------------------------------------------------------------------------ }
{                                                                          }
{ - Die maximale Lnge fr Header und 'encoded words' liegt in dieser      }
{   Routine bei jeweils 65500 Zeichen (ein 'encoded word' darf laut        }
{   RFC2047, Section 2, nur max. 75 Zeichen lang sein).                    }
{ - Die Lnge gefoldeter Headerzeilen, die encoded words enthalten, ist    }
{   in dieser Routine beliebig (RFC2047: max. 76 Zeichen), da der Header   }
{   bereits in entfalteter Form vorliegt, wenn er hier bearbeitet wird.    }
{ - Wenn 'rfcMime' auf false (= Default) steht, nimmt die Routine im Sinne }
{   des "robustness principle" eine extrem fehlertolerante Decodierung vor }
{   und decodiert im Unterschied zu bisher z.B. auch encoded words, die    }
{   Leerzeichen enthalten oder sonstwie verunstaltet sind (was beides in   }
{   der Realitt leider auch vorkommt).                                    }
{ - Steht 'rfcMime' jedoch auf true, verhlt sich die Routine quasi wie    }
{   ein Checkbot, beachtet die im jeweiligen Kontext (strukturierter/      }
{   unstrukturierter Header, quoted-string, Kommentar usw.) geltenden      }
{   unterschiedlichen Regeln und decodiert keine Strings, die im           }
{   jeweiligen Kontext kein gltiges 'encoded word' bilden.                }
{ - 'encoded words' mit einem ungltigen oder von XP nicht untersttzten   }
{   Zeichensatz und/oder einer ungltigen Codierungskennung werden nicht   }
{   mehr decodiert (weil sie nicht decodiert werden knnen) - Delimiter,   }
{   Charset-Deklaration und Codierung bleiben dann vollstndig erhalten    }
{   (bisher wurden sie vernichtet).                                        }
{ - Im unwahrscheinlichen Fall, da ein Header lnger als 65500 Zeichen    }
{   ist und sich am Ende ein unvollstndiges 'encoded word' ohne gltigen  }
{   Abschlu-Delimiter befindet, decodieren wir dennoch den Teil des       }
{   'encoded word', der sich noch innerhalb des Arrays befindet, und       }
{   hngen in 'WriteHeader' bei unstrukturierten Headern oder bei als      }
{   Backup angelegten U-Zeilen zur Kennzeichnung fr die Krzung die       }
{   Zeichenkette '[...]' an.                                               }
{ - Inkomplette encoded words am Ende des Headers, bei denen schon der     }
{   Anfangs-Delimiter unvollstndig ist (z.B. in der Charset-Deklaration   }
{   abgeschnitten), werden als "dubios" verworfen.                         }
{ - Im Falle, da es sich um einen nicht MIME-codierten Header handelt,    }
{   wird eine ISO1=>IBM-Konvertierung durchgefhrt (weil es leider doch    }
{   immer wieder Header mit uncodierten 8bit-Zeichen gibt, und das sind    }
{   i.d.R. Header im ISO1/15- bzw. im verwandten Win-1252-Zeichensatz).    }
{ - Ist der Parameter 'replaceWSP' true, werden TABs durch Leerzeichen     }
{   ersetzt. Gibt der Parameter 'context' gleichzeitig an, da es sich um  }
{   einen strukturierten Header handelt, dann werden auch (uncodierte!)    }
{   Mehrfach-Leerzeichen, die sich auerhalb eines Kommentars und quoted-  }
{   strings befinden, durch ein Leerzeichen ersetzt. 'replaceWSP' ist      }
{   eigentlich nur fr den erstmaligen Aufruf von 'GetHeader' in der UUZ-  }
{   Routine 'ReadRFCheader' gedacht, durch die *alle* Header laufen - wo   }
{   aber noch keine MIME-Decodierung stattfindet. Er sollte bei MIME-Deco- }
{   dierung (Parameter 'MIMEdecode') mit 'false' bergeben werden, da er   }
{   dort unntig ist und nur Laufzeit kosten wrde.                        }
{                                                                          }
{ ber den Wrapper 'RFC2047_Decode' kann diese Routine auch fr Strings    }
{ eingesetzt werden ('replaceWSP' steht dann festverdrahtet auf 'false').  }
{ ------------------------------------------------------------------------ }

{ context: 0 = unstrukturierter Header/Teilheader           }
{          1 = strukturierter Header/Teilheader             }
{          2 = wie 1, aber als Kommentar behandeln, obwohl  }
{              keine Klammern vorhanden (fr Realnames!)    }
{                                                           }
{          (wirkt sich auf die Behandlung von encoded words }
{          und das Entfernen mehrfacher Leerzeichen aus)    }

function GetHeader(var hdr:LongHdrP; var len:word;
                   var rest:string; var truncated:boolean;
                   const overflow,MIMEdecode,replaceWSP:boolean;
                   const context:byte):string;

var      i,p0,p1,p2,openc,              { openc   = Anz. offene Klammern,  }
         klammer,klammer2  : word;      { klammer = Pos. (nchste) Klammer }
                deleteWSP,              { Leerzeichen vor nchstem EW lschen }
                   isMIME,              { Header ist MIME-codiert }
     possibleIncompleteEW,              { mgliches inkomplettes EW }
                incomment,              { in Kommentar      }
                   inquote : boolean;   { in quoted-string  }
                    qchars : string[3]; { quoted-pair-Zeichen }
                         s : string;

  procedure checkCommentQuote(const hdrpos:word);

    function foundClose(c:word; const x:char):word;
      function QuotedChar:boolean;
      var q : word;
      begin
        q:=c;
        while (q>1) and (hdr^[q-1]='\') do dec(q);
        QuotedChar:=odd(c-q);
      end;
    begin
      foundClose:=0;
      while (c<len) and not          { schlieende(s) DQUOTE/Klammer suchen }
            ((hdr^[c]=x) and not QuotedChar)
        do inc(c);
      if (hdr^[c]=x) and not QuotedChar then
        foundClose:=c
      else if overflow then
        foundClose:=len+1;
    end;

  begin
    case hdr^[hdrpos] of
      '"' : if not incomment {!} then
              if not inquote then
                inquote:=foundClose(hdrpos+1,'"')>0
              else
                inquote:=false;  { dann mu es ein schlieendes DQUOTE sein }
      '(' : if not inquote {!} then
              if not incomment then
              begin
                klammer:=foundClose(hdrpos+1,')');
                if klammer>0 then
                begin
                  incomment:=true;
                  openc:=1;
                end;
              end
              else begin
                klammer2:=foundClose(klammer+1,')');
                if klammer2>0 then
                begin
                  inc(openc);
                  klammer:=klammer2;
                end;
              end;
      ')' : if not inquote {!} and incomment then
            begin
              if openc>0 then dec(openc);
              if openc=0 then
              begin
                incomment:=false;
                klammer:=0;
              end;
            end;
    end;
  end;

  procedure DecodeHeader(const CheckIncompleteEW:boolean);
  type listeP = ^listeT;
       listeT = record
                  p1        : word;     { erstes  Zeichen des EW   }
                  p2        : word;     { letztes Zeichen des EW   }
                  deleteWSP : boolean;  { WSP links vom EW lschen }
                  next      : listeP;
                end;

  var   validEW : boolean;
             cs : string[30]; { charset  }
           code : char;       { encoding }
             ii : word;       { Universalvariable, wird mehrfach benutzt! }
     start,lauf : listeP;
   nonMIMEchars : string[50]; { nicht erlaubte Zeichen in EWs }
  label next,lexit;

  const    CTLs = #0#1#2#3#4#5#6#7#8#9#10#11#12#13+        { Control-Chars  }
                  #14#15#16#17#18#19#20#21#22#23#24+       { plus '?', nie  }
                  #25#26#27#28#29#30#31#32#127+'?';        { erlaubt in EWs }

   specials2822 = '()<>@,;:\".[]';       { RFC2822 specials, Teilmenge von: }
      especials = '/=';                     { zustzliche RFC2047 especials }

    nonQinComms = '()\';           { bei '?Q?' in Kommentaren nicht erlaubt }
   nonQinQuotes = '"\';         { bei '?Q?' in quoted-strings nicht erlaubt }

  { --------------------------------------------------------- }
  { Hier angewandte Regeln fr gltige encoded words          }
  { (wenn 'rfcMime' auf true steht):                          }
  { --------------------------------------------------------- }
  { - In 'charset' und 'encoding' drfen nicht vorkommen:     }
  {   CTLs + specials2822 + especials                         }
  {                                                           }
  { - In 'encoded text' darf nicht vorkommen:                 }
  {   a) Bei 'Q' encoding:                                    }
  {      aa) In strukturierten Headern (From:, To:, Cc:, ...) }
  {          aaa) CTLs + nonQinComms, wenn in Kommentar       }
  {          aab) CTLs + nonQinQuotes, wenn in quoted-string  }
  {               (encoded words sind in quoted-strings       }
  {               eigentlich gar nicht erlaubt, werden hier   }
  {               aber toleriert und decodiert)               }
  {          aac) CTLs + specials2822, wenn auerhalb         }
  {               Kommentar und EWs als 'atoms' auftreten     }
  {      ab) In unstrukturierten Headern (Subject): CTLs      }
  {   b) Bei 'B' encoding:                                    }
  {      CTLs + specials2822 (wird hier aber immer decodiert, }
  {                           das mu der Decoder abfangen)   }
  { --------------------------------------------------------- }

    procedure DecodeEW;
    const trailing_WSPs : word = 0;
    var pe2,n,anfang : word;                { pe2 = nchste Startposition-1 }
                long,                      { unvollstndiges bzw. langes EW }
         is_last_wsp,          { (wenn EW < 256, kann 's' nie > 248 werden) }
        was_last_wsp : boolean;
    begin
      code:=' ';
      ii:=(p2-p1)+1;                       { Lnge des zu decodierenden EWs }
      long:=ii>255;
      is_last_wsp:=false;          { true, wenn WSP im (letzten) Teilstring }
      was_last_wsp:=false;         { durch Lschen von CR|LF entstanden ist }
      cs_fallback:=false;
      s[0]:=chr(iif(long,maxUTFLen,ii));
      fastmove(hdr^[p1],s[1],byte(s[0]));                         { EW => s }
      if deleteWSP then                  { WSPs zwischen EWs "entfernen"... }
      begin
        while (p0>1) and (hdr^[p0-1]=' ') do dec(p0);
        inc(p0,trailing_WSPs);    { ... aber keine WSPs *im* vorherigen EW! }
      end;
      anfang:=p0;           { Anfangsposition des *decodierten* EW im Array }
      n:=cpos('?',mid(s,4));                         { Pos. des zweiten '?' }
      if n=0 then exit;                   { darf eigentlich nicht vorkommen }
      cs:=lstr(copy(s,3,n));                             { Charset   merken }
      fcs:=fallback_charset(convertible_charset(cs));    { Fallback-Charset }
      code:=upcase(s[n+4]);                              { Codierung merken }
      s:=mid(s,n+6);                          { Anfangs-Delimiter entfernen }
      n:=0;           { Lnge der Delimiter fr 1. Durchlauf initialisieren }
      pe2:=p1-1;                   { nchste Startposition-1 initialisieren }
      start_of_UTF:=true;  { evtl. 'sc_rest' bei UTF-Decodierung ignorieren }
      { ------------------------------------------------------------------- }
      repeat
        pe2:=iif(long,pe2+min(p2-pe2,maxUTFLen-n),p2);  { nchste Startp.-1 }
        if not CheckIncompleteEW and (p2<=pe2) then
          dec(byte(s[0]),2);                { ggf. '?=' am Ende abschneiden }
        if n=0 then
          n:=byte(cs[0])+7;                          { Lnge des Delimiters }
        case code of
          'Q' : begin
                  if long or CheckIncompleteEW then
                    if s[byte(s[0])-1]='=' then    { endet z.B. mit '=E4=E' }
                    begin
                      dec(byte(s[0]),2);                    { s um 2 krzen }
                      if not CheckIncompleteEW and (p2>pe2) then
                        dec(pe2,2);    { long: neue nchste Startposition-1 }
                    end
                    else if (s[byte(s[0])]='=')     { endet z.B. mit '=E4=' }
                         or (s[byte(s[0])]='?')     { endet z.B. mit '=E4?' }
                    then begin
                      dec(byte(s[0]));                      { s um 1 krzen }
                      if not CheckIncompleteEW and (p2>pe2) then
                        dec(pe2);      { long: neue nchste Startposition-1 }
                    end;
                  repeat                                   { Q-Decodierung: }
                    ii:=cpos('_',s);
                    if ii>0 then s[ii]:=' ';                 { '_' ersetzen }
                  until ii=0;
                  UnQuotePrintable(s,0);
                end;
          'B' : begin
                  if long or CheckIncompleteEW then
                  begin
                    ii:=byte(s[0]) mod 4;
                    dec(byte(s[0]),ii);
                    if not CheckIncompleteEW and (p2>pe2) then
                      dec(pe2,ii);     { long: neue nchste Startposition-1 }
                  end;
                  DecodeBase64(s);                          { B-Decodierung }
                end;
        end;
        end_of_UTF:=p2<=pe2;                 { 'sc_rest' nicht mehr belegen }
        if fcs='' then                 { kein Fallback-Charset vorhanden => }
          CharsetToIBM(cs,s);          { non-Fallback-Charsets konvertieren }
        start_of_UTF:=false;              { 'sc_rest' nicht mehr ignorieren }
        ii:=0;
        while ii<length(s) do                       { #0 und <TAB> ersetzen }
        begin
          inc(ii);
          if s[ii] in [#0,#9] then s[ii]:=' ';
        end;
        is_last_wsp:=false;
        repeat
          ii:=cpos(#10,s);                                  { <LF> ersetzen }
          if (ii>0) then
          begin
            if ((ii>1) and (s[ii-1] in [' ',#13])) or
               ((ii=1) and ((p0=1) or ((p0>1) and (hdr^[p0-1]=' ')))) or
               ((ii<byte(s[0])) and (s[ii+1] in [' ',#13])) or
               ((ii=byte(s[0])) and (p2<=pe2)) then
              delete(s,ii,1)
            else begin
              s[ii]:=' ';
              is_last_wsp:=ii=byte(s[0]);
            end;
          end;
        until ii=0;
        repeat
          ii:=cpos(#13,s);                                  { <CR> ersetzen }
          if (ii>0) then
          begin
            if ((ii>1) and (s[ii-1]=' ')) or
               ((ii=1) and ((p0=1) or ((p0>1) and (hdr^[p0-1]=' ')))) or
               ((ii<byte(s[0])) and (s[ii+1]=' ')) or
               ((ii=byte(s[0])) and (p2<=pe2)) then
              delete(s,ii,1)
            else begin
              s[ii]:=' ';
              is_last_wsp:=is_last_wsp or (ii=byte(s[0]));
            end;
          end;
        until ii=0;
        { Wenn der letzte Teilstring deswegen mit einem WSP  }
        { endete, weil ein CR|LF durch ein WSP ersetzt wurde }
        { und wenn der aktuelle Teilstring ebenfalls mit     }
        { einem WSP beginnt, mu dieses entfernt werden (wir }
        { konnten das erste Zeichen des nchsten Teilstrings }
        { nicht "sehen", als CR|LF durch WSP ersetzt wurde). }
        if was_last_wsp and (firstchar(s)=' ') then
          delfirst(s);
        was_last_wsp:=is_last_wsp;
        repeat
          ii:=cpos(#26,s);                              { <Ctrl-Z> ersetzen }
          if ii>0 then s[ii]:='>';
        until ii=0;
        if (fcs<>'') and not cs_fallback then           { Charset-Fallback? }
        begin
          ii:=0;
          repeat                                    { bei erstem gefundenen }
            inc(ii);                                { Zeichen abbrechen und }
            cs_fallback:=byte(s[ii]) in [$80..$9f]; { Routine nicht wieder  }
          until cs_fallback or (ii=byte(s[0]));     { aufrufen              }
        end;
        fastmove(s[1],hdr^[p0],byte(s[0]));            { decod. EW => Array }
        inc(p0,byte(s[0]));               { p0 = Position+1, bis zu der das }
        if pe2<p2 then                    {      Array fertig decodiert ist }
        begin
          ii:=min(p2-pe2,maxUTFLen-n);
          fastmove(hdr^[pe2+1],s[1],ii);        { nchster Teil des EW => s }
          s[0]:=chr(ii);
        end
      until (p2<=pe2);                        { Ende der Decodierung des EW }
      { ------------------------------------------------------------------- }
      if fcs<>'' then         { Fallback-Charsets (ISO-8859-x) konvertieren }
      begin
        n:=anfang;
        ii:=(p0-n);                  { Lnge des zu konvertierenden Strings }
        while ii>0 do
        begin
          s[0]:=chr(iif(ii>255,255,ii));
          fastmove(hdr^[n],s[1],byte(s[0]));                  { String => s }
          if cs_fallback then cs:=fcs;
          CharsetToIBM(cs,s);          { Charset-Konvertierung (ISO-8859-x) }
          fastmove(s[1],hdr^[n],byte(s[0]));     { konvert. String => Array }
          dec(ii,byte(s[0]));    { Lnge des zu konvertierenden Reststrings }
          inc(n,byte(s[0]));               { nchste Startposition im Array }
        end;
      end;
      { Um beim evtl. Entfernen von WSPs *zwischen* dem aktuellen }
      { und dem nchsten EW (codierte) WSPs *innerhalb* des       }
      { aktuellen EW nicht mit zu entfernenden WSPs auerhalb     }
      { desselben zu verwechseln, merken wir uns, ob und wieviele }
      { "echte" trailing blanks der Gesamtheader aktuell hat.     }
      n:=p0;
      while (n>1) and (hdr^[n-1]=' ') do dec(n);
      trailing_WSPs:=p0-n;
      if (lauf=nil) or                   { => letztes gltiges EW im Array! }
         CheckIncompleteEW then
      begin
        n:=len-p2;
        len:=(p0+n)-1;                                { Lnge aktualisieren }
        if not CheckIncompleteEW then
        begin
          p1:=p0-1;                      { p0 sichern fr CheckIncompleteEW }
          deleteWSP:=true;               { (ab p0 auf IncompleteEW checken) }
        end;
      end                     { Lnge des ggf. nach links zu verschiebenden }
      else n:=(lauf^.p1-p2)-1;      { Strings bis zum nchsten EW ermitteln }
      if (n>0) and not CheckIncompleteEW then
      begin                    { String von p2+1 bis nchstes EW nach links }
        move(hdr^[p2+1],hdr^[p0],n);
        inc(p0,n);                        { p0 = Position+1, bis zu der das }
      end;                                {      Array fertig decodiert ist }
      start_of_UTF:=false;  { Default setzen! }
      end_of_UTF:=false;    { Default setzen! }
    end;   { of DecodeEW }

  begin   { of DecodeHeader }          { gesamtes Array nach gltigen EWs   }
    start:=nil;                        { durchkmmen und Start-/Endposition }
    validEW:=false;                    { jedes EW sowie WSP-Flag (lschen   }
    possibleIncompleteEW:=false;       { ja/nein) in verketteter Liste      }
    cs:=''; code:=' ';                 { speichern                          }
    if not CheckIncompleteEW {!} then    { bei CheckIncompleteEW sind diese }
    begin                                { Variablen vorher gesetzt worden! }
      deleteWSP:=false;
      inquote:=false;
      if context=2 then
      begin
       incomment:=true;
       openc:=1;
       klammer:=len+1;
      end
      else begin
        incomment:=false;
        openc:=0;
        klammer:=0;
      end;
      p1:=0;
    end;
    p0:=0; p2:=0;
    nonMIMEchars:=CTLs + specials2822 + especials;   { fr charset/encoding }
    { --------------------------------------------------------------------- }
    repeat                                        { '=?' vor Charset suchen }
      p1:=min(p1+1,len);
      if p1>len-1 then goto next;
      if not ((hdr^[p1]='=') and (hdr^[p1+1]='?')) then
      begin
        if context>0 then                           { strukturierter Header }
        begin
          if (hdr^[p1]='\') and (p1<len) and (inquote or incomment) then
          begin
            if inquote then qchars:='\"' else
            if incomment then qchars:='\()';
            if cpos(hdr^[p1+1],qchars)>0 then                   { skip next }
            begin
              inc(p1);
              goto next;
            end;
          end
          else checkCommentQuote(p1);
        end;
        if hdr^[p1]<>' ' then      { Zeichen <> ' ' zwischen zwei Words:    }
          deleteWSP:=false;        { => keine zwei EWs direkt nebeneinander }
        continue;                  { => WSP nicht entfernen                 }
      end
      else if (p1>1) and rfcMime and not               { vor EW prfen auf: }
              ((hdr^[p1-1]=' ') or                     { Leerzeichen oder   }
               (incomment and (hdr^[p1-1]='('))) then  { '(' in Kommentar   }
        continue;
      possibleIncompleteEW:=overflow;           { '=?' vor Charset gefunden }
      p0:=p1+1;
      ii:=p1-1;                  { p1-1 sichern fr evtl. CheckIncompleteEW }
                                 { (ab p1 auf IncompleteEW checken)         }
      { ------------------------------------------------------------------- }
      repeat                             { encoding '?Q?' oder '?B?' suchen }
        p0:=min(p0+1,len);
        if p0>len-2 then
        begin
          if CheckIncompleteEW then
          begin
            p2:=len;
            validEW:=true;
            goto lexit;
          end
          else begin                      { dann kann's kein EW mehr werden }
            p1:=p0;
            goto next;
          end;
        end
        else if not ((hdr^[p0]='?') and (hdr^[p0+2]='?')) then
        begin
          if cpos(hdr^[p0],nonMIMEchars)>0 then       { unerlaubtes Zeichen }
          begin                                       { im Charset gefunden }
            deleteWSP:=false;                         { => kein gltiges EW }
            possibleIncompleteEW:=false;
            p1:=p0;
            goto next;
          end;
          if context>0 then                         { strukturierter Header }
            checkCommentQuote(p0);
          continue;                               { nchstes Zeichen prfen }
        end;
        p2:=(p0-p1)-2;                { 'p2' fr Lnge von 'cs' mibrauchen }
        if p2<sizeof(cs) then
        begin
          fastmove(hdr^[p1+2],cs[1],p2);                  { Charset => 'cs' }
          cs[0]:=chr(p2);
          cs:=lstr(cs);
        end
        else cs:='';
        code:=upcase(hdr^[p0+1]);     { encoding '?Q?' oder '?B?' gefunden? }
        if not ((code in ['Q','B']) and (cs<>'') and
                supported_charset(cs)) then
        begin
          deleteWSP:=false;
          possibleIncompleteEW:=false;
          p1:=p0+2;
          goto next;
        end;
        p2:=p0+2;
        { ----------------------------------------------------------------- }
        repeat                                               { encoded text }
          p2:=min(p2+1,len);
          if p2>len-1 then
          begin
            if CheckIncompleteEW then
            begin
              p2:=len;
              validEW:=true;
              goto lexit;
            end
            else begin
              p1:=p2;
              goto next;
            end;
          end
          else if not (((hdr^[p2]='?') and (hdr^[p2+1]='=')) and
                       (not rfcMime or 
                        (rfcMime and
                         ((p2+1=len) or
                          ((p2+1<len) and
                           ((hdr^[p2+2]=' ') or
                            ((context=1) and
                             (cpos(hdr^[p2+2],specials2822)>0)))))))) then
          begin
            if rfcMime and (code='Q') then case context of   { 'Q' encoding }
                0 : nonMIMEchars:=CTLs;           { unstrukturierter Header }
              1,2 : if incomment then             { strukturierter   Header }
                      nonMIMEchars:=CTLs + nonQinComms
                    else if inquote then
                      nonMIMEchars:=CTLs + nonQinQuotes
                    else
                      nonMIMEchars:=CTLs + specials2822;
            end
            else                          { 'B' encoding oder 'not rfcMime' }
              nonMIMEchars:='';           { => immer decodieren!            }
            if (cpos(hdr^[p2],nonMIMEchars)>0) and { unerlaubtes Zeichen in }
               not (overflow and not truncated and { encoded text gefunden  }
                   (p2=len) and (hdr^[p2]='?') and { => kein gltiges EW    }
                   (left(rest,1)='=')) then        { ('?' als letztes       }
            begin                                  { Zeichen abfangen)      }
              deleteWSP:=false;
              possibleIncompleteEW:=false;
              p1:=p2;
              goto next;
            end;
            if context>0 then                       { strukturierter Header }
              checkCommentQuote(p2);
            continue;                             { nchstes Zeichen prfen }
          end;
          inc(p2);                      { '?=' (Ende encoded text) gefunden }
          validEW:=true;
          possibleIncompleteEW:=false;
        until validEW or (p2>len-1);
      until validEW or (p0>len-1) or (p2>len-1);
  lexit:
      if validEW then
      begin
        if start=nil then
        begin
          new(start);
          start^.next:=nil;
          start^.p1:=p1;
          start^.p2:=p2;
          start^.deleteWSP:=deleteWSP;
          lauf:=start;
        end
        else begin           { hinten anfgen }
          new(lauf^.next);
          lauf:=lauf^.next;
          lauf^.next:=nil;
          lauf^.p1:=p1;
          lauf^.p2:=p2;
          lauf^.deleteWSP:=deleteWSP;
        end;
        isMIME:=true;
        validEW:=false;
        deleteWSP:=true;              { 'deleteWSP' gilt fr *nchstes* EW! }
        p1:=p2;
      end;
  next:
    until (p0>len-1) or (p1>len-1) or                 { Ende der EW-Analyse }
          (p2>len-1);
    { --------------------------------------------------------------------- }
    if start<>nil then
    begin
      lauf:=start;
      p0:=start^.p1;                      { p0 = Position+1, bis zu der das }
      while lauf<>nil do                  {      Array fertig decodiert ist }
      begin
        p1:=lauf^.p1;
        deleteWSP:=lauf^.deleteWSP;
        p2:=lauf^.p2;
        lauf:=lauf^.next;
        if CheckIncompleteEW and not (code in ['Q','B']) then
        begin
          len:=p0-1;                 { unvollstndiges EW dubios, wegwerfen }
          truncated:=true;
        end else
          DecodeEW;
      end;
      while assigned(start) do                 { verkettete Liste freigeben }
      begin
        lauf:=start^.next;
        dispose(start);
        start:=lauf;
      end;
    end
    else if possibleIncompleteEW then
      p1:=ii;    { gesichertes p1-1 fr evtl. CheckIncompleteEW zurckholen }
  end;   { of DecodeHeader }

begin   { of GetHeader }

  GetHeader:='';
  if len=0 then exit;
  isMIME:=false;
  inquote:=false;
  if context=2 then
  begin
    incomment:=true;
    openc:=1;
    klammer:=len+1;
  end
  else begin
    incomment:=false;
    openc:=0;
    klammer:=0;
  end;

  if replaceWSP then                    { TABs und ggf. Multi-WSPs ersetzen }
  begin
    while (len>0) and (hdr^[len] in [' ',#9]) do  { rechte WSPs 'entfernen' }
      dec(len);    
    if len=0 then exit;
    i:=0;
    while i<len do
    begin
      inc(i);
      if hdr^[i]=#9 then hdr^[i]:=' ';                      { TABs ersetzen }
      if (context>0) then                          { strukturierter Header: }
      begin 
        if (hdr^[i]='\') and (i<len) and (inquote or incomment) then
        begin
          if inquote then qchars:='\"' else
          if incomment then qchars:='\()';
          if cpos(hdr^[i+1],qchars)>0 then                      { skip next }
          begin
            inc(i);
            continue;
          end;
        end
        else if hdr^[i]=' ' then
        begin
          if not (inquote or incomment) then
          begin
            p0:=i;
            while (p0<len) and                          { Multi-WSPs suchen }
                  (hdr^[p0] in [' ',#9]) and
                  (hdr^[p0+1] in [' ',#9]) do
              inc(p0);
            if p0>i then                              { Multi-WSPs gefunden }
            begin
              move(hdr^[p0+1],hdr^[i+1],len-p0);     { Multi-WSPs entfernen }
              dec(len,p0-i);                               { Lnge anpassen }
            end;
          end;
        end
        else checkCommentQuote(i);
      end;
    end;
    incomment:=false;
    inquote:=false;
    openc:=iif(context=2,1,0);
  end;

  if MIMEdecode then                                   { MIME-Decodierung }
  begin
    deleteWSP:=false;
    DecodeHeader(false);
    if possibleIncompleteEW then
    begin
      DecodeHeader(true);       { unvollstndig codierten Rest decodieren }
      rest:='';            { Rest-Zeichen (max. 4) > 65500 nicht anhngen }
    end;
    if overflow and                    { Sonderfall: ' =' am Ende, knnte }
       (not possibleIncompleteEW) and  { auch ein inkomplettes EW sein    }
       (hdr^[len]='=') and (not rfcMime or
       ((hdr^[len-1]=' ') or
        ((context>0) and (hdr^[len-1]='(')))) and
       (left(rest,1)='?') then
    begin
      dec(len);
      if deleteWSP then
        while hdr^[len]=' ' do dec(len);
      rest:='';
    end;
    if not isMIME then                      { wenn nicht MIME-codiert,    }
      IsoToIBM(hdr^,len,cs_win1252);        { ISO1/Win=>IBM-Konvertierung }
  end;

  i:=min(len,255);
  fastmove(hdr^[1],s[1],i);                                  { Array => s }
  s[0]:=chr(i);
  GetHeader:=s;

end;   { of GetHeader }


{ my: Wrapper fr MIME-Decodierung von Strings via 'GetHeader' 09/2002 }

{ context: 0 = unstrukturierter Header }
{          1 = strukturierter Header   }
{          2 = wie 1, aber Kommentar   }

procedure RFC2047_Decode(var s:string; const context:byte);
var   sp : LongHdrP;
    slen : byte;       { Lnge des bergebenen Strings }
    dlen : word;       { Wrapper-Dummy }
       r : string[1];  { Wrapper-Dummy }
       t : boolean;    { Wrapper-Dummy }
begin
  if s='' then exit;
  sp:=nil;
  slen:=length(s);
  dlen:=slen;
  r:='';
  t:=false;
  if maxavail>=slen then
  begin
    getmem(sp,slen);
    fastmove(s[1],sp^[1],slen);
    s:=GetHeader(sp,dlen,r,t,false,                { MIME-Decodierung, ver- }
                 true,false,context);              { ndert evtl. 'dlen'    }
    freemem(sp,slen);
  end;
end;


{ ------------------------------------------------------------------------ }
{ my: 'RfcToZcConv' - Neue Routine 08/2002-04/2003                         }
{                                                                          }
{     Multifunktionsroutine als Ersatz fr das frhere 'GetAdr', die ber  }
{     Adressen und Realnames hinaus jetzt auch Newsgroups, MsgIDs und      }
{     diverse andere RFC-Header in eine ZC-gerechte Form bringt, dabei ggf.}
{     via 'GetHeader' MIME-decodiert sowie Kommentare und berflssige bzw.}
{     falsche quoted-pairs/strings korrekt entfernt (und je nach Header    }
{     korrekt gesetzte quoted-pairs/strings beibehlt). Das Verhalten der  }
{     Routine im Detail wird ber den Parameter 'context' gesteuert.       }
{                                                                          }
{     Diese Routine ist ausschlielich zur Bearbeitung strukturierter      }
{     Header, die IDs (Adressen, Newsgroups, MsgIDs etc.) enthalten,       }
{     gedacht und geeignet (also z.B. nicht fr 'Keywords:')!              }
{                                                                          }
{     Erheblich verbessertes, RFC2822-konformeres Parsing sowie Entschr-  }
{     fung von Limits: Bei der Bearbeitung werden Header mit einer Gesamt- }
{     lnge von bis zu 65500 Zeichen in einem dynamischen Array vom Typ    }
{     'LongHdrP' untersttzt und das Ergebnis dann als fertige Strings an  }
{     die Parameter 'str1' bzw. 'str2' zurckgegeben, wobei 'str2' aus-    }
{     schlielich dem Realname vorbehalten ist, da bei allen Headern, die  }
{     keine Adressen/Realnames enthalten, nur die Rckgabe eines einzelnen }
{     Strings (eben an 'str1') erwartet wird.                              }
{                                                                          }
{     Die gesamte Behandlung findet direkt innerhalb des 64k-Arrays statt, }
{     so da keine Fehlinterpretationen durch die Stringlngenbegrenzung   }
{     auf 255 Zeichen entstehen.                                           }
{                                                                          }
{     Soweit im Einzelfall mglich, wird bei Headern, die lnger als 65500 }
{     Zeichen sind, der sich noch innerhalb des Arrays befindliche Teil    }
{     des Headers korrekt identifiziert und behandelt.                     }
{                                                                          }
{     Fr die Konvertierung von Strings kann diese Routine ber den        }
{     Wrapper 'RFC2822_Remove' aufgerufen werden, wobei im Falle von       }
{     Strings, die Adressen und Realnames enthalten, nur die bearbeitete   }
{     Adresse (nicht aber der Realname) zurckgegeben wird.                }
{                                                                          }
{ ------------------------------------------------------------------------ }
{                                                                          }
{ Die Header werden im einzelnen der folgenden Behandlung unterzogen:      }
{                                                                          }
{                                                                          }
{ 1. Adressen/Newsgroups:                                                  }
{ -----------------------                                                  }
{                                                                          }
{ Aus einem komma-separierten RFC-Header wird die eMail-Adresse und der    }
{ Realname bzw. die Newsgroup des ersten Elements der Liste ermittelt.     }
{ Enthlt die Liste mehrere Adressen oder Newsgroups, wird das bearbeitete }
{ Element aus dem Array nach links rausgeschoben und es befindet sich das  }
{ nchste Element am Anfang des Arrays (so knnen alle Adressen/Newsgroups }
{ ber einen Schleifenaufruf ermittelt werden).                            }
{                                                                          }
{ Adressen/Realnames/Newsgroups werden im Detail wie folgt konvertiert:    }
{                                                                          }
{ Adressen:                                                                }
{ ---------                                                                }
{ - Group-Adressierung ("A Group: a@b.de, c@d.de;") wird entfernt. Leere   }
{   Groups wie "Undisclosed Recipients:;" werden dabei aufgelst zu:       }
{   "!Empty_group@Undisclosed_Recipients"                                  }
{ - Kommentare in Adressen werden entfernt (<theo(bla)@test.de(blubb)>     }
{   wird jetzt korrekt aufgelst zu "theo@test.de").                       }
{ - Leerzeichen in Adressen (<theo . test @ test . de>) werden entfernt,   }
{   sofern sie sich nicht im quoted-string eines local part befinden.      }
{ - Route-Adressen (<@machine1.tld,@machine2.tld:mary@example.net>) werden }
{   korrekt entfernt (laut RFC2822 sind Route-Adressen eigentlich obsolet, }
{   mssen bei eingehenden Nachrichten aber weiterhin als zulssig behan-  }
{   delt werden).                                                          }
{ - UUCP-"Bang"-Adressierung wird wie bisher in eine Domain-Adresse umge-  }
{   wandelt: "kiste1!kiste2!test.de!theo" wird zu "theo@test.de"           }
{ - Bei Adressen, deren local part unntigerweise als quoted-string        }
{   vorliegt und/oder berflssige quoted-pairs enthlt, werden diese      }
{   entfernt (<"th\eo"@test.de> wird zu "theo@test.de" aufgelst).         }
{ - Wenn sich nach der Auflsung eine leere Adresse ergibt, setzt XP die   }
{   ber den Parameter 'emptyfiller' bergebene Dummy-Adresse ein. Ist die }
{   Adresse nur deshalb leer, weil zwei Kommata unmittelbar aufeinander    }
{   folgen, wird sie komplett ignoriert.                                   }
{ - An nicht-leere Adressen ohne Domain (knnen z.B. entstehen, wenn der   }
{   local part lnger als die ber den Parameter 'len1' bergebene Lnge   }
{   ist) wird die Domain "@??" angehngt.                                  }
{                                                                          }
{ Ansonsten bleiben Adressen (speziell hinsichtlich korrekt gesetzter und  }
{ notwendiger quoted-pairs/strings) unverndert, um zu gewhrleisten, da  }
{ eine Antwort exakt an die Adresse in der Form erfolgt, wie sie der       }
{ Absender angegeben hat.                                                  }
{                                                                          }
{ Realnames:                                                               }
{ ----------                                                               }
{ - Der Realname (nicht die Adresse!) wird via 'GetHeader' MIME-decodiert  }
{   (siehe Kommentar vor 'GetHeader').                                     }
{ - Kommentare, Anfhrungszeichen (DQUOTEs) und quoted-pairs im Realname   }
{   werden jetzt vollstndig und den Regeln entsprechend (!) entfernt.     }
{                                                                          }
{   Beispiel:                                                              }
{                                                                          }
{     From: =?'.'?B?"\\\"?=" "\\                                           }
{      \"blickst Du noch durch ? ich nicht" <test@it.example>              }
{                                                                          }
{     Daraus machte XP bzw. UUZ bisher:                                    }
{       ABS: test@it.example (\\")       *Argh*                            }
{     Richtig ist:                                                         }
{       ABS: test@it.example (=?'.'?B?\"?= \ "blickst Du [...] ich nicht)  }
{                                                                          }
{   (Hier spielt allerdings auch die bisher fehlerhafte MIME-Decodierung   }
{   eine wesentliche Rolle, siehe Kommentar vor 'GetHeader'.)              }
{                                                                          }
{ Newsgroups:                                                              }
{ -----------                                                              }
{ - Punkte werden durch Slashes ersetzt.                                   }
{ - Kommentare und Leerzeichen werden entfernt (obgleich beides dort nicht }
{   vorkommen drfte).                                                     }
{                                                                          }
{                                                                          }
{ 2. MsgIDs:                                                               }
{ ----------                                                               }
{                                                                          }
{ Im Unterschied zu Adressen/Newsgroups geht die Routine bei MsgIDs davon  }
{ aus, da es sich nicht um eine komma-separierte Liste, sondern entweder  }
{ um eine einzelne oder mehrere MsgIDs handelt, die Phrasen und/oder       }
{ quoted-strings auerhalb sowie Kommentare in- und auerhalb von MsgIDs   }
{ enthalten knnen. Die Parameter 'len1' (max. zulssige Stringlnge fr   }
{ 'str1') und 'context' sind entsprechend zu bergeben; es wird dann im    }
{ Prinzip nur die Subroutine 'RFC2822_CleanupAdr' ausgefhrt und die erste }
{ (oder einzige) nach den untenstehenden Regeln gefundene MsgID an 'str1'  }
{ zurckgegeben und wie bei Adressen/Newsgroups nach links aus dem Array   }
{ rausgeschoben, so da in einem Schleifenaufruf die weiteren MsgIDs       }
{ ermittelt werden knnen (z.B. fr "References:").                        }
{                                                                          }
{ - Kommentare und Leerzeichen werden (jetzt korrekt) entfernt.            }
{ - berflssige/falsche quoted-pairs/strings werden entfernt.             }
{ - Wegen der obs-Regeln in RFC2822 werden Phrasen und quoted-strings      }
{   zwischen MsgIDs jetzt ignoriert.                                       }
{                                                                          }
{ Ist kein in spitze Klammern eingeschlossener String auerhalb von        }
{ Kommentaren und quoted-strings vorhanden, geben wir einen Leerstring an  }
{ 'str1' zurck. Eine Prfung auf Gltigkeit der MsgID findet (bewut)     }
{ nicht statt.                                                             }
{                                                                          }
{                                                                          }
{ 3. Sonstige Header:                                                      }
{ -------------------                                                      }
{                                                                          }
{ Bei Headern, die nicht zu einem der o.g. gehren, wird im Prinzip nur    }
{ die Subroutine 'RFC2822_Cleanup' ausgefhrt, d.h., es werden Kommentare  }
{ und (optional) jetzt auch quoted-pairs/strings korrekt und vollstndig   }
{ entfernt (und damit das frhere und nicht korrekt arbeitende             }
{ 'RfcRemoveComment' ersetzt).                                             }
{                                                                          }
{ Ist zustzlich eine MIME-Decodierung dieser Header erforderlich, ist     }
{ diese ber einen zustzlichen direkten Aufruf von 'GetHeader' oder - im  }
{ Falle von Strings - ber den Wrapper 'RFC2047_Decode' zu realisieren.    }
{                                                                          }
{ ------------------------------------------------------------------------ }

{ context: 0 = Header/Teilheader mit einer/mehreren Mail-Adressen  }
{          1 = Header/Teilheader mit einer/mehreren Newsgroups     }
{          2 = Header/Teilheader mit einer/mehreren MsgIDs         }
{          3 = sonstiger Header (Kommentare und Quoting entfernen) }
{          4 = sonstiger Header (nur Kommentare entfernen)         }

procedure RfcToZcConv(var hdr:LongHdrP; var len:word; const len1,len2:word;
                      var str1,str2,rest:string; var truncated:boolean;
                      const overflow:boolean; context:byte;
                      const emptyfiller:string);
var p,p2,pk,openc,
    a,mark1,mark2,
  prebang,bangpos,
   dompos,klammer,
          klammer2 : word;
             lastc : char;
          inquote,              { in quoted-string         }
        addrstart,              { Start '<>'-Adresse       }
        angleaddr,              { ist '<>'-Adresse         }
       legacyaddr,              { ist '@ ()'-Adresse       }
           inaddr,              { in '<>'-Adresse          }
         indomain,              { in Domain                }
          literal,              { in Domain-Literal        }
        incomment,              { in Kommentar             }
          ingroup,              { in Group-Adressierung    }
        emptyaddr,              { leere Adresse (z.B. <> ) }
       emptygroup,              { leere Group-Adresse      }
             atend : boolean;   { am Headerende angekommen }
            qchars : string[3]; { quoted-pair-Zeichen      }

  { String von 'von' bis 'bis' entfernen und Lnge/Position aktualisieren }

  procedure removeString(const von:word; bis:word);
  begin
    if bis>len then exit;                 { zur Sicherheit }
    move(hdr^[bis+1],hdr^[von],len-bis);  { String entfernen }
    bis:=(bis-von)+1;                     { Lngendifferenz ermitteln }
    dec(len,bis);                         { Headerlnge aktualisieren }
    dec(pk,bis);                          { Headerposition aktualisieren }
  end;

  { UUCP-"Bang"-Adressierung auflsen - nur nach RFC2822_CleanupAdr aufrufen }
  { (sonst sind 'prebang', 'bangpos' und 'dompos' nicht korrekt gesetzt)!    }

  procedure resolveBang(const a:word; var b:word);
  var l : word;
  begin
    if emptygroup or (prebang+bangpos=0) or
       (bangpos>b) then exit;
    if dompos=0 then                                        { keine Domain: }
    begin
      if prebang>a then                    { mehrere Bang-Systeme angegeben }
      begin
        l:=(prebang-a)+1;
        removeString(a,prebang);   { alle Systeme bis auf letztes entfernen }
        dec(bangpos,l);
        dec(b,l);
        prebang:=0;
      end;
      l:=b-bangpos;
      move(hdr^[bangpos+1],hdr^[bangpos],l);   { Systemnamen zeichenweise   }
      while bangpos>a+1 do                     { hinter Usernamen schaufeln }
      begin
        hdr^[b]:=hdr^[a];
        move(hdr^[a+1],hdr^[a],b-a);
        dec(bangpos);
      end;
      hdr^[b]:=hdr^[a];
      move(hdr^[a+1],hdr^[a],l);
      hdr^[a+l]:='@';
    end
    else begin                                          { Domain vorhanden: }
      removeString(a,bangpos);                { alle Bang-Systeme entfernen }
      dec(b,(bangpos-a)+1);
    end;
  end;

  function foundClose(c:word; const ende:word; const x:char):word;
    function QuotedChar:boolean;
    var q : word;
    begin
      q:=c;
      while (q>1) and (hdr^[q-1]='\') do dec(q);
      QuotedChar:=odd(c-q);
    end;
  begin
    foundClose:=0;
    while (c<ende) and not           { schlieende(s) DQUOTE/Klammer suchen }
          ((hdr^[c]=x) and not QuotedChar)
      do inc(c);
    if (hdr^[c]=x) and not QuotedChar then
      foundClose:=c
    else if overflow and (ende=len) then
      foundClose:=ende+1;
  end;

  { 'removeThis' arbeitet hnlich wie 'RemoveString',  }
  { aktualisiert aber zustzlich die Endposition des   }
  { zu bearbeitenden Teilheaders sowie die aktuelle    }
  { Position innerhalb dieses Teilheaders (sofern 's'  }
  { als zweiter Parameter bergeben wurde).            }
  { Diese Routine ist nur zum Aufruf aus Unterroutinen }
  { heraus geeignet, die wiederum aus der Hauptroutine }
  { aufgerufen wurden (z.B. 'RFC2822_CleanupAdr'!      }

  procedure removeThis(var von,bis,ende:word);
  begin
    if bis>len then exit;     { zur Sicherheit }
    removeString(von,bis);    { remove comment, WSP, '\', '"', '<' or '>' }
    dec(ende,(bis-von)+1);    { update end position }
    bis:=von-1;               { reset current position }
  end;

  procedure trimArray(var anfang,ende:word);
  var p1,p2 : word;
  begin
    p1:=ende;                                                   { righttrim }
    p2:=ende;
    while (p1>0) and (hdr^[p1]=' ') do dec(p1);
    if p1<ende then
    begin
      inc(p1);
      removeThis(p1,p2,ende);
    end;
    anfang:=min(anfang,ende);
    p1:=anfang;                                                  { lefttrim }
    p2:=anfang;
    while (p2>0) and (hdr^[p2]=' ') do inc(p2);
    if p2>anfang then
    begin
      dec(p2);
      removeThis(p1,p2,ende);
    end;
  end;

  { Entfernt RFC-Kommentare, WSPs und berflssige (!) quoted-pairs in  }
  { Adressen und MsgIDs, ignoriert bzw. beachtet dabei auch quoted-     }
  { strings und bercksichtigt die Regeln fr quoted-pairs; im Sinne    }
  { des "robustness principle" werden auch ungltige Kommentare         }
  { entfernt, die sich *in* einem Atom befinden.                        }
  {                                                                     }
  { Bei MsgIDs wird in dieser Routine die gefundene MsgID aus dem Array }
  { nach links "rausgeschoben", um ggf. weitere MsgIDs in einem         }
  { Schleifenaufruf ermitteln zu knnen (fr Adressen findet das in der }
  { Hauptroutine statt).                                                }
  {                                                                     }
  { Diese Routine setzt auch 'prebang', 'bangpos' und 'dompos' fr      }
  { 'resolveBang'.                                                      }
  {                                                                     }
  { Im Falle von Newsgroups werden hier nur Kommentare und Leerzeichen  }
  { entfernt (das Ersetzen von Punkten durch Slashes findet in der      }
  { Hauptroutine statt).                                                }
  {                                                                     }
  { Anmerkung zur Behandlung von MsgIDs:                                }
  { ------------------------------------                                }
  { Wie schon bisher, werden auch in MsgIDs Kommentare entfernt (jetzt  }
  { allerdings auch in allen Fllen korrekt), und darber hinaus jetzt  }
  { auch berflssige/falsche quoted-pairs/strings sowie unzulssige    }
  { Leerzeichen. Damit ist gewhrleistet, da syntaktisch unterschied-  }
  { liche, aber semantisch identische MsgIDs in XP auch als identische  }
  { MsgIDs behandelt werden.                                            }
  { Allerdings ist dieses Verfahren nach den RFC-Gepflogenheiten nicht  }
  { ganz sauber (und es war nie sauber), weil MsgIDs prinzipiell nicht  }
  { verndert werden sollen. Es ist daher zu berlegen, ob nicht lang-  }
  { fristig diese MsgID-Behandlung zur Laufzeit in XP selbst statt im   }
  { UUZ vorgenommen werden sollte.                        (my 08/2002)  }

  procedure RFC2822_CleanupAdr(var anfang,ende:word; const RemoveWSP:boolean);

  var    s,p,c,q,             { c=comment, q=quoted-string,                 }
       oMid,cMid : word;      { in MsgID: oMid='<', cMid='>' of first MsgID }
             d,l,             { d=in-domain, l=literal,                     }
           qs_ok,             { quoted-string needed,                       }
         doneMid : boolean;   { MsgID found                                 }
  const sc = #9#32+'()<>[]:;@,"';   { RFC2822 special chars - OK in quoted- }
             { without dot due }    { strings w/o quoted-pair, tokenization }
             { to dot-atoms!   }    { points elsewhere (we ignore comments  }
                                    { here as they are being removed anyway)}

  begin
    s:=anfang-1;
    p:=0; c:=0; q:=0;
    oMid:=0; cMid:=0;
    d:=false;
    l:=false;
    qs_ok:=false;
    doneMid:=false;
    emptyaddr:=angleaddr;
    while (s<ende) and (len>0) and not ((context=2) and doneMid) do
    begin
      inc(s);
      if emptyaddr then emptyaddr:=hdr^[s]=' ';  { leere angleaddr abfangen }
      case hdr^[s] of
        '\' : if (context in [0,2]) and
                 (s<ende) and ((q>0) or (c>0) or l) then
              begin
                if q>0 then qchars:='\"' else           { remove superfluos }
                if c>0 then qchars:='\()' else          { quoted-pairs      }
                if l then qchars:='\[]';
                if cpos(hdr^[s+1],qchars)=0 then   { superfluos quoted-pair }
                  removeThis(s,s,ende)
                else begin                           { quoted-pair required }
                  if q>0 then qs_ok:=true;
                  inc(s);                                       { skip next }
                end;
              end;
        '"' : if (context in [0,2]) and (c=0) and (dompos=0) then
                if q=0 then
                begin
                  if foundClose(s+1,ende,'"')>0 then
                    q:=s
                end
                else begin
                  if not qs_ok then                        { remove DQUOTEs }
                  begin
                    removeThis(s,s,ende);             { remove last  DQUOTE }
                    removeThis(q,q,ende);             { remove first DQUOTE }
                    dec(s);           { update position due to first DQUOTE }
                  end;
                  q:=0;
                end;
        '!' : if not ((q>0) or (c>0) or (context<>0))    { UUCP 'bang' path }
                 and (dompos=0)
              then begin
                if bangpos>0 then prebang:=bangpos;
                bangpos:=s;
              end;
        '@' : if (context=0) and (q=0) then
              begin
                d:=not ((c>0) or l);                   { set in-domain flag }
                if d and (dompos=0) then dompos:=s;
              end
              else qs_ok:=true;
        '[' : if (context=0) and (q=0) then
              begin
                if d and not l then l:=true;         { set literal flag on  }
              end
              else qs_ok:=true;
        ']' : if (context=0) and (q=0) then
              begin
                if d and l then l:=false;            { set literal flag off }
              end
              else qs_ok:=true;
        '(' : if not ((q>0) or l) then
              begin
                if c=0 then
                begin
                  klammer:=foundClose(s+1,ende,')');
                  if klammer>0 then
                  begin
                    c:=1;
                    p:=s;                                { start of comment }
                  end;
                end
                else begin
                  klammer2:=foundClose(klammer+1,ende,')');
                  if klammer2>0 then
                  begin
                    inc(c);                             { inc comment count }
                    klammer:=klammer2;
                  end;
                end;
              end
              else qs_ok:=true;
        ')' : if not ((q>0) or l) then
              begin
                if c=1 then
                begin
                  removeThis(p,s,ende);                    { remove comment }
                  c:=0;                               { reset comment count }
                  klammer:=0;
                end
                else if c>0 then dec(c);                { dec comment count }
              end
              else qs_ok:=true;
        '<' : if (context=2) and not                       { start of MsgID }
                 ((q>0) or (c>0)) then
                oMid:=s                       { remember '<' of first MsgID }
              else qs_ok:=true;
        '>' : if (context=2) and (oMid>0) and not
                 ((q>0) or (c>0)) then
              begin
                cMid:=s;                      { remember '>' of first MsgID }
                doneMid:=true;
              end
              else qs_ok:=true;
        ' ' : if (RemoveWSP and not ((q>0) or (c>0))) or
                 ((context=2) and (c=0)) then
              begin
                p:=s;
                while (s<ende) and (hdr^[s+1]=' ') do
                  inc(s);
                removeThis(p,s,ende);                         { remove WSPs }
              end
              else if q>0 then qs_ok:=true;
        else if not qs_ok and (context in [0,2]) and (q>0) and
             (cpos(hdr^[s],sc)>0) then
          qs_ok:=true;
      end;  { case }
    end;
    if (context=2) then                                   { determine MsgID }
    begin
      if cMid>oMid then                           { we have a MsgID '<...>' }
      begin
        p:=cMid;
        ende:=cMid;
        removeThis(cMid,p,ende);                          { remove '>', ... }
        p:=anfang;
        removeThis(p,oMid,ende);      { ... '<' and everything before MsgID }
      end
      else begin                                 { we have no MsgID '<...>' }
        oMid:=anfang;
        cMid:=ende;
        removeThis(oMid,cMid,ende);
      end;
    end;
    if ende>0 then trimArray(anfang,ende);
  end;

  { (Teil-)Header vollstndig (!) von Kommentaren, Multi-WSPs }
  { und quoted-pairs/strings befreien (z.B. bei Realnames)    }
  {                                                           }
  { Gibt 'false' zurck, wenn ein "alter" Realname in Quotes  }
  { stand und daher als "neuer" Realname erkannt wurde, der   }
  { offenbar von einem Mail/News-Gate o.. fehlerhaft in die  }
  { alte Form umgesetzt wurde.                                }

  function RFC2822_Cleanup(var anfang,ende:word; oldRn:boolean):boolean;
  var s,p,c,q : word; { c=comment, q=quoted-string }
  begin
    RFC2822_Cleanup:=true;
    if oldRn then
    begin
      if (hdr^[anfang]='(') and     { Klammern bei altem Realname entfernen }
         (hdr^[ende]=')') then      { (damit nicht der Realname insgesamt   }
      begin                         { als Kommentar entsorgt wird)          }
        p:=ende;
        removeThis(p,p,ende);
        p:=anfang;
        removeThis(p,p,ende);
        trimArray(anfang,ende);
      end;
      if (hdr^[anfang]='"') and      { Sonderfallbehandlung:                }
         (hdr^[ende]='"') then       { Bei Anfhrungszeichen vor/nach altem }
      begin                          { Realname auf das Vorkommen von Vor-  }
        p:=anfang+1;                 { und Nachname prfen und im weiteren  }
        s:=ende-1;                   { ggf. wie neuen Realname behandeln    }
        while (p<ende-1) and              { erstes Nicht-Leerzeichen suchen }
              (hdr^[p]=' ') do
          inc(p);
        if (p<ende-1) and              { nchstes Leerzeichen danach suchen }
           (hdr^[p]<>' ') then
        begin
          s:=p+1;
          while (s<ende-1) and
                (hdr^[s]<>' ') do
            inc(s);
        end
        else if (hdr^[p]=' ') then            { nur Leerzeichen zwischen "" }
          p:=ende;
        if (p<ende-1) and (s<ende-1) and       { nchstes Nicht-Leerzeichen }
           (hdr^[s]=' ') then                  { suchen, um "Nicknames"     }
        begin                                  { unverndert zu lassen      }
          p:=s+1;
          while (p<ende-1) and
                (hdr^[p]=' ') do inc(p);
          if p=ende-1 then                { kein Nicht-Leerzeichen gefunden }
            p:=anfang;
        end;
        if p>s then                   { "normaler" Realname mit Leerzeichen }
        begin                         { oder "" oder nur Leerzeichen in ""  }
          oldRn:=false;
          RFC2822_Cleanup:=false;
        end;
      end;
    end;
    s:=anfang-1;
    p:=0; c:=0; q:=0;
    while (s<ende) and (len>0) do
    begin
      inc(s);
      case hdr^[s] of
        '\' : if (s<ende) and                         { remove quoted-pairs }
                 ((q>0) or (c>0) or oldRn) then
                begin
                  if q>0 then qchars:='\"' else
                  if (c>0) or oldRn then qchars:='\()';
                  if context<>4 then
                    removeThis(s,s,ende);
                  if cpos(hdr^[s+1],qchars)>0 then { if quoted-pair...      }
                    inc(s);                        { ...required, skip next }
                end;
        '"' : if not ((c>0) or oldRn) then
                if q=0 then
                begin
                  if foundClose(s+1,ende,'"')>0 then
                    q:=s
                end
                else begin
                  if context<>4 then
                  begin
                    removeThis(s,s,ende);             { remove last  DQUOTE }
                    removeThis(q,q,ende);             { remove first DQUOTE }
                    dec(s);           { update position due to first DQUOTE }
                  end;
                  q:=0;
                end;
        '(' : if q=0 then
                if c=0 then
                begin
                  klammer:=foundClose(s+1,ende,')');
                  if klammer>0 then
                  begin
                    c:=1;
                    p:=s;                                { start of comment }
                  end;
                end
                else begin
                  klammer2:=foundClose(klammer+1,ende,')');
                  if klammer2>0 then
                  begin
                    inc(c);                             { inc comment count }
                    klammer:=klammer2;
                  end;
                end;
        ')' : if q=0 then
                if c=1 then
                begin
                  removeThis(p,s,ende);                    { remove comment }
                  c:=0;                               { reset comment count }
                  klammer:=0;
                end
                else if c>0 then dec(c);                { dec comment count }
        ' ' : if not ((c>0) or (q>0)) and
                 (hdr^[s-1]=' ') then
                removeThis(s,s,ende);    { remove Multi-WSPs between tokens }
      end;  { case }
    end;
    if ende>0 then trimArray(anfang,ende);
  end;

  { Adresse/Newsgroup/MsgID konvertieren und nach 'str1' kopieren }

  procedure copyAdr(var a,b:word);
  begin
    if (context=0) and ((b<a) or                    { Sonderflle abfangen, }
       ((a=b) and                                   { z.B. '<>', '>', '\'   }
        (hdr^[a] in ['"','>','\','(',')',',','@','.']))) then
      b:=0
    else
      RFC2822_CleanupAdr(a,b,not emptygroup);{ Kommentare/Quoting entfernen }
    if (b=0) or
        ((context=0) and                        { Kommt z.B. vor, wenn die }
         (emptyaddr or                          { Adresse nur aus '< >',   }
          (angleaddr and                        { '""' o.. bestand        }
           (b<=a) and (hdr^[a] in ['<','>'])))) then
      str1:=''
    else begin
      if context=0 then resolveBang(a,b);       { UUCP-"Bang"-Adressierung? }
      p2:=min((b-a)+1,len1);           { 'p2' fr Lnge Adresse mibrauchen }
      fastmove(hdr^[a],str1[1],p2);
      str1[0]:=chr(p2);
      str1:=trim(str1);
      if context=1 then                            { Newsgroups: '.' => '/' }
      begin
        repeat
          p2:=cpos('.',str1);
          if p2>0 then str1[p2]:='/';
        until p2=0;
        str1:='/'+str1;
      end
      else if emptygroup then                         { '!Empty_group@str1' }
      begin
        repeat
          p2:=cpos(' ',str1);
          if p2>0 then str1[p2]:='_';
        until p2=0;
        str1:='!Empty_group@'+str1;
      end;
    end;
  end;

  { Realname konvertieren und nach 'str2' kopieren      }
  { (diese Routine immer nur *nach* 'copyAdr' aufrufen, }
  { weil sie davon ausgeht, da sich die Adresse nicht  }
  { mehr im Array befindet!)                            }

  procedure CopyRealname(var a,b:word; old:boolean);
  begin
    while (b>1) and (hdr^[b]=' ') do dec(b);         { rechte Leerzeichen }
    if b<pk then removeString(b+1,pk);               { entfernen          }
    old:=RFC2822_Cleanup(a,b,old) and old; { Kommentare/Quoting entfernen }
    { ------------------------------------------------------------------- }
    GetHeader(hdr,b,rest,truncated,                   { MIME-Decodierung, }
              overflow,true,false,byte(old)+1);       { verndert evtl. b }
    if b<pk then                  { Realname war codiert...               }
      if not atend then           { nicht am Ende des Headers angekommen: }
        removeString(b+1,pk)      { Zeichenberhang entfernen             }
      else begin
        pk:=b; len:=b;            { am Ende des Headers angekommen        }
      end;
    { ------------------------------------------------------------------- }
    p2:=min(pk,len2);               { 'p2' fr Lnge Realname mibrauchen }
    fastmove(hdr^[1],str2[1],p2);           { Realname => 'str2' kopieren }
    str2[0]:=chr(p2);
    if (pk>len2) or (atend and truncated) then
      str2:=left(str2,len2-5)+'[...]';
    str2:=trim(str2);
  end;

  { Array vom nchsten Nicht-WSP nach 'n' an nach    }
  { links schieben und Lnge/Position aktualisieren  }

  procedure moveToLeft(n:word);
  begin
    while (n<len) and (hdr^[n+1]=' ') do        { WSPs nach 'n' entfernen }
      inc(n);
    dec(len,n);
    dec(pk,n);
    if len>0 then
      move(hdr^[n+1],hdr^[1],len);             { Rest nach links schieben }
  end;

begin   { of RfcToZcConv }
  if len=0 then exit;
  str1:=''; str2:='';
  inquote:=false;
  addrstart:=false;
  angleaddr:=false;
  legacyaddr:=false;
  inaddr:=false;
  indomain:=false;
  literal:=false;
  incomment:=false;
  ingroup:=false;
  emptyaddr:=false;
  emptygroup:=false;
  atend:=false;
  lastc:=' ';
  p:=0; pk:=0; openc:=0; klammer:=0; mark1:=0; mark2:=0;
  prebang:=0; bangpos:=0; dompos:=0; 
  if context in [2,3,4] then           { MsgID oder sonstiger (Teil-)Header }
    pk:=len
  else repeat                                { Adresse(n) oder Newsgroup(s) }
    inc(pk);
    if (hdr^[pk]=',') and (pk=1) then          { Komma am Anfang ignorieren }
    begin                                      { (bei mehreren Kommata un-  }
      moveToLeft(pk);                          { mittelbar hintereinander)  }
      pk:=0;
      if len=0 then exit;                     { wenn Header mit Komma endet }
    end
    else case hdr^[pk] of                                   { Mail-Adresse: }
            { ---------------------------------------------------- }
            { quoted-pair ist erlaubt:                             }
            { - in Kommentaren und quoted-strings generell         }
            {   (auch z.B. innerhalb des Realname);                }
            { - im local part, wenn er als quoted-string vorliegt; }
            { - in der Domain, wenn sie als Literal vorliegt.      }
            { ---------------------------------------------------- }
      '\' : if (inquote or incomment or literal) and            { skip next }
               (pk<len) then inc(pk);
      '"' : if not incomment {!} then     { quoted-strings in Kommentaren   }
              if not inquote then         { gelten nicht als quoted-strings }
                inquote:=foundClose(pk+1,len,'"')>0
              else
                inquote:=false;  { dann mu es ein schlieendes DQUOTE sein }
      '<' : if not (angleaddr or inquote or incomment or inaddr) then
            begin
              addrstart:=true;
              inaddr:=true;
              angleaddr:=true;
              mark1:=pk;                     { Startposition Adresse merken }
            end;
      '>' : if inaddr and not (inquote or incomment or literal) then
            begin
              if ingroup then angleaddr:=false; { bei ingroup zurcksetzen! }
              inaddr:=false;
              literal:=false;
              indomain:=false;
              mark2:=pk;                       { Endposition Adresse merken }
            end;
      '@' : if addrstart and not   { RFC822-Route-Adresse? ('@' ist erstes  }
               ingroup then        { Zeichen, nur in '<>'-Adresse erlaubt!) }
            begin
              addrstart:=false;
              p:=pk;
              while (p<len) and (pk>mark1) and not
                    ((hdr^[p]='>') and not incomment) do
              begin
                inc(p);
                if hdr^[p]='(' then        { Der zu findende ':' darf nicht }
                begin                      { in einem Kommentar stehen      }
                  if not incomment then
                  begin
                    klammer:=foundClose(p+1,len,')');
                    if klammer>0 then
                    begin
                      incomment:=true;
                      openc:=1;
                    end;
                  end
                  else begin
                    klammer2:=foundClose(klammer+1,len,')');
                    if klammer2>0 then
                    begin
                      inc(openc);
                      klammer:=klammer2;
                    end;
                  end;
                end
                else if (hdr^[p]=')') and incomment then
                begin
                  if openc>0 then dec(openc);
                  if openc=0 then
                  begin
                    incomment:=false;
                    klammer:=0;
                  end;
                end
                else if (hdr^[p]=':') and not incomment     { Route-Adresse }
                then begin                                  { entfernen     }
                  removeString(mark1+1,p);
                  pk:=mark1;          { Position resetten, Schleife beenden }
                end;
              end;
            end                                           { Beginn Domain   }
            else begin
              indomain:=inaddr and not                    { in '<>'-Adresse }
                        (inquote or incomment or literal);
              legacyaddr:=(context in [0,1]) and not
                          (angleaddr or inquote or incomment or literal);
            end;
      '[' : if indomain and not (inquote or incomment or literal) then
              literal:=true;
      ']' : if literal and not (inquote or incomment) then
              literal:=false;
      '(' : if not inquote {!} then          { Kommentare in quoted-strings }
              if not incomment then          { gelten nicht als Kommentare  }
              begin
                klammer:=foundClose(pk+1,len,')');
                if klammer>0 then
                begin
                  incomment:=true;
                  openc:=1;
                  if not angleaddr then
                  begin
                    mark1:=pk;             { Startposition Kommentar merken }
                    mark2:=0;
                  end;
                end;
              end
              else begin
                klammer2:=foundClose(klammer+1,len,')');
                if klammer2>0 then
                begin
                  inc(openc);
                  klammer:=klammer2;
                end;
              end;
      ')' : if not inquote {!} then          { Kommentare in quoted-strings }
            begin                            { gelten nicht als Kommentare  }
              if incomment then
              begin
                dec(openc);
                if openc=0 then
                begin
                  incomment:=false;
                  klammer:=0;
                end;
              end;
              if not angleaddr then          { Endposition Kommentar merken }
                mark2:=pk;
            end;
      ':' : if not (inquote or inaddr or incomment) then
            begin
              p:=pk;            { evtl. Group-Adressierung, Pos. ':' merken }
              p2:=pk+1;
              while (p2<len) and (hdr^[p2]=' ') do
                inc(p2);                              { Beginn Group merken }
              pk:=p2-1;
              ingroup:=true;
            end;
      ';' : if ingroup and not               { Group-Adressierung entfernen }
               (inquote or inaddr or incomment) then
            begin
              ingroup:=false;
              if (pk>p2) then
              begin
                if pk=len then
                  dec(len)
                else
                  hdr^[pk]:=',';
                removeString(1,p2-1);
                pk:=0;
                lastc:=' ';
                continue;              { Group-Adressen nochmal durchlaufen }
              end
              else begin                               { pk=p2, leere Group }
                emptygroup:=true;
                hdr^[pk]:=',';
                removeString(p,pk-1);
                pk:=p;
              end;
            end;
      else if ingroup and (pk=len) then                       { keine Group }
      begin
        inquote:=false;
        addrstart:=false;
        inaddr:=false;
        indomain:=false;
        literal:=false;
        incomment:=false;
        ingroup:=false;
        pk:=p;
        continue;                       { Header ab ':' nochmal durchlaufen }
      end
      else if addrstart then               { wenn erstes Nicht-WSP nach '<' }
        addrstart:=hdr^[pk]=' ';           { <> '@', nicht mehr auf Route-  }
    end;  { case }                         { Adresse bei case '@' prfen    }
    if not (hdr^[pk] in [',',' ']) then         { letztes "echtes" Zeichen  }
      lastc:=hdr^[pk];                          { ungleich Komma/WSP merken }
  until ((pk>0) and (hdr^[pk]=',') and not
        (inquote or inaddr or incomment or ingroup)) or (pk>=len);

  if (context=1) and (angleaddr or legacyaddr) then { Adresse in Newsgroup? }
    context:=0;

  if context in [0,1] then                  { Mail-Adressen oder Newsgroups }
  begin
    atend:=pk>=len;
    if atend then
    begin
      if not (inquote or inaddr or incomment) and (hdr^[pk]=',') then
      begin
        dec(len);
        pk:=len;
      end;
    end
    else dec(pk);                   { 'pk' auf letztes Zeichen vor ',' setzen }
  end;

  p:=pk;
  a:=1;

  if context in [1,2] then                         { Newsgroups oder MsgID: }
    copyAdr(a,p)                                      { verndert evtl. 'p' }

  else if context in [3,4] then                  { sonstiger (Teil-)Header: }
  begin
    RFC2822_Cleanup(a,p,false);       { Kommentare, quoted-pairs/strings    }
                                      { usw. entfernen, verndert evtl. 'p' }
    p2:=min(p,len1);
    fastmove(hdr^[1],str1[1],p2);               { Header => 'str1' kopieren }
    str1[0]:=chr(p2);
  end

  else begin                                    { context=0 (Mail-Adresse): }
    if not angleaddr then       { alte Form: local.part@do.main (Real Name) }
    begin
      if (openc>0) and (hdr^[mark1-1]=' ') and        { ')' fehlt, evtl.    }
         atend and overflow then                      { abgeschnitten o..? }
      begin
        inc(len);
        inc(pk);
        inc(p);
        hdr^[pk]:=')';                                 { evtl. ')' ergnzen }
        lastc:=')';
        mark2:=pk;
        openc:=0;
      end;
      if (mark1>0) and (mark2>mark1) then
      begin
        if ((openc=0) and (lastc=')')) then           { Realname vorhanden? }
        begin
          p:=mark1-1;
          while (p>1) and (hdr^[p]=' ') do   { rechte Leerzeichen berlesen }
            dec(p);
          copyAdr(a,p);    { Adresse => 'str1', verndert evtl. 'a' und 'p' }
          moveToLeft(p);                      { Adresse aus Array entfernen }
          p:=pk;
          a:=min(1,p);
          copyRealname(a,p,true);{ Realname => 'str2', verndert evtl. 'len' }
        end
        else copyAdr(a,p);{ kein Realname, sondern Kommentar, hdr => 'str1' }
      end 
      else copyAdr(a,p);                { gar kein Kommentar, hdr => 'str1' }
    end
    else begin                  { neue Form: Real Name <local.part@do.main> }
      if (mark2<=mark1) then                                { '>' fehlt...? }
      begin
        mark2:=pk;
        p:=mark2;
      end
      else p:=mark2-1;
      a:=mark1+1;
      copyAdr(a,p);        { Adresse => 'str1', verndert evtl. 'a' und 'p' }
      if mark1>1 then                                 { Realname vorhanden? }
      begin
        removeString(mark1,pk);               { Adresse aus Array entfernen }
        p:=pk;
        a:=min(1,p);
        copyRealname(a,p,false);{ Realname => 'str2', verndert evtl. 'len' }
      end;
    end;
  end;

  if context in [0,1] then                  { Mail-Adressen oder Newsgroups }
  begin
    if not atend then inc(pk);                        { 'pk' auf ',' setzen }
    moveToLeft(pk);                              { Rest nach links schieben }

    if (context=0) and not (emptygroup or emptyaddr) then
      if str1='' then str1:=emptyfiller
      else if (cpos('@',str1)=0) and          { kommt z.B. bei local parts  }
              (firstchar(str1)<>'/') then     { grer 'AdrLen' vor         }
      begin
        truncstr(str1,len1-3);
        while lastchar(str1)='.' do
          dellast(str1);
        str1:=str1+'@??';
      end;
  end
  else if context=2 then                                           { MsgIDs }
  begin
    pk:=p;
    moveToLeft(pk);                              { Rest nach links schieben }
  end;

end;   { of RfcToZcConv }


{ my: Wrapper zur RFC=>ZC-Konvertierung von Strings }
{     via 'RfcToZcConv'           09/2002 + 04/2004 }

procedure RFC2822_Remove(var s:string; const context:byte);
var   sp : LongHdrP;
    slen : byte;       { Lnge des bergebenen Strings }
    dlen : word;       { Wrapper-Dummy }
       r : string[1];  { Wrapper-Dummy }
       t : boolean;    { Wrapper-Dummy }
begin
  sp:=nil;
  slen:=length(s);
  dlen:=slen;
  r:='';
  t:=false;
  if maxavail>=slen then
  begin
    getmem(sp,slen);
    fastmove(s[1],sp^[1],slen);                  { Kommentare/Quoting in    }
    RfcToZcConv(sp,dlen,slen,slen,s,r,r,t,       { String entfernen, vern- }
                false,context,'');               { ndert evtl. 'dlen'      }
    freemem(sp,slen);
  end;
end;


function MakeMimeBoundary:string;
var i : byte;
    s : string[100];
begin
  s:='';
  for i:=1 to 10+random(20) do
    s:=s+char(random(25)+byte('A'));
  s:='-==_'+xp_display+'_Next_MIME_Part_'+s+'_==-';
  MakeMimeBoundary:=s;
end;


{ -------------------------------------------------------------- }
{ my: Absolute Position eines Strings in einem langen Header vom }
{     Typ LongHdrP ab Position zurckgeben (0 = nicht gefunden)  }
{                                                                }
{     'frompos' ist die Stelle, ab der gesucht werden soll,      }
{     'topos' die Stelle, bis zu der gesucht werden soll (i.d.R. }
{     die Lnge des Headers, es knnen so aber auch nur Teilbe-  }
{     reiche des Headers durchsucht werden).                     }
{ -------------------------------------------------------------- }
function posLong(const s:string; const hdr:LongHdrP; frompos:word;
                 const topos:word; const docase:boolean):word;
var found : boolean;
       p2 : word;
begin
  posLong:=0;
  if frompos=0 then frompos:=1;
  if (length(s)>0) and (topos>0) and (frompos<=topos) and
     (length(s)<=(topos-frompos)+1) then
  begin
    dec(frompos);
    found:=false;
    while not found and (frompos<=topos-(length(s))) do
    begin
      inc(frompos);
      found:=true;
      p2:=0;
      while found and (p2<length(s)) do                   { Stringvergleich }
      begin
        inc(p2);
        if docase then
          found:=s[p2]=hdr^[frompos+(p2-1)]                { case-sensitive }
        else
          found:=LoCase(s[p2])=LoCase(hdr^[frompos+(p2-1)]);  { ignore case }
      end;
    end;
    if found then posLong:=frompos;
  end;
end;

{ vollst?ndige RFC-1522-Decodierung }

procedure MimeIsoDecode(var ss:string; maxlen:integer);
var p1,p2,p,i : integer;
    lastEW,
    nextW     : integer;
    code      : char;
    s         : string;
    cset      : string[20];
begin
  for i:=1 to length(ss) do
    if ss[i]=#9 then ss[i]:=' ';

  cset:='';
  p1:=0;
  lastEW:=0;
  repeat
    repeat
      p1:=posn('=?',ss,p1+1);
      if p1>0 then begin
        p2:=p1+2;
        i:=0;
        while (i<3) and (p2<length(ss)) do begin
          if ss[p2]='?' then inc(i)
          else if ss[p2]=' ' then break;
          inc(p2);
        end;
        if (i<3) or (ss[p2]<>'=') then p2:=0 else dec(p2);
      end;
    until (p1=0) or (p2>0);

    if (p1>0) and (p2>0) then begin
      if (lastEW>0) and (lastEW<nextW) and (p1=nextW) then begin
        nextW:=nextW-lastEW;
        delete(ss,lastEW,nextW);
        dec(p1,nextW);
        dec(p2,nextW);
      end;
      s:=copy(ss,p1+2,p2-p1-2);
      delete(ss,p1,p2-p1+2);
      p:=cpos('?',s);
      if p>0 then begin
        cset:=lstr(left(s,p-1));
        delete(s,1,p);
        p:=cpos('?',s);
        if p=2 then begin
          code:=UpCase(s[1]);
          delete(s,1,2);
          case code of
            'Q' : begin
                    for i:=1 to length(s) do
                      if s[i]='_' then s[i]:=' ';
                    s:=s+'=';
                    UnquotePrintable(s,0);
                  end;
            'B' : UnquotePrintable(s,0);

          end;
        end;
      end;
      CharsetToIBM(cset,s);
      insert(s,ss,p1);
      lastEW:=p1+length(s);
      nextW:=lastEW;
      while (nextW<length(ss)) and (ss[nextW]=' ') do inc(nextW);
    end;
  until (p1=0) or (p2=0);

  if length(ss)>maxlen then ss[0]:=char(maxlen);
  if cset='' then ISO2IBM(ss,cs_iso8859_1);  { ISO-Decode wenn kein RFC1522 }
  for i:=1 to length(ss) do
    if ss[i]<' ' then ss[i]:=' ';
end;



end.
